!    COLOR FUNCTIONS (routines for Colorhelix & TestRGBtable)
!    Copyright (C) 2011,2012 by Ingo Thies
!
!    Color Functions is free software: you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by
!    the Free Software Foundation, either version 2 of the License, or
!    (at your option) any later version.
!
!    Color Functions is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with Color Functions. If not, see <http://www.gnu.org/licenses/>.
!
MODULE color_functions_mathconst
  DOUBLE PRECISION :: pi6=0.52359877559829887308d0
  DOUBLE PRECISION :: pi4=0.78539816339744830962d0
  DOUBLE PRECISION :: pi3=1.04719755119659774615d0
  DOUBLE PRECISION :: pi2=1.57079632679489661923d0
  DOUBLE PRECISION :: pi =3.14159265358979323846d0
  DOUBLE PRECISION :: tau=6.28318530717958647693d0
  DOUBLE PRECISION :: r2d=5.72957795130823208768d1
  DOUBLE PRECISION :: d2r=1.74532925199432957692d-2
  DOUBLE PRECISION :: r2g=6.36619772367581343076d1
  DOUBLE PRECISION :: g2r=1.57079632679489661923d-2
  DOUBLE PRECISION :: third=1.d0/3.d0
  DOUBLE PRECISION :: ten2nine=10.d0/9.d0
END MODULE color_functions_mathconst

!     Matrices are of dimention 3x3 of the following form:
!     /a11 a12 a13\   /array(1,1) array(2,1) array(3,1)\
!     |a21 a22 a23| = |array(1,2) array(2,2) array(3,2)|
!     \a31 a32 a33/   \array(1,3) array(2,3) array(3,3)/
!
!     Note that array(j,i) = aij (left entry revolves fastest)

MODULE color_functions
  INTEGER :: cf_verbose=0
  INTEGER,PARAMETER :: CF_SINGLE=4
  INTEGER,PARAMETER :: CF_DOUBLE=8
! from http://en.wikipedia.org/wiki/LMS_color_space (02.09.2011)
! and http://www.babelcolor.com/download/A%20review%20of%20RGB%20color%20spaces.pdf
  REAL(KIND=CF_DOUBLE) :: krgb2xyz=0.17697d0
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_d50=(/0.96422d0,1.00000d0,0.82521d0/)!sum=2.78943
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_d55=(/0.95682d0,1.00000d0,0.92149d0/)!sum=2.87831
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_d65=(/0.95047d0,1.00000d0,1.08883d0/)!sum=3.03930
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_d75=(/0.94972d0,1.00000d0,1.22638d0/)!sum=3.17610
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_9300k=(/0.97135d0,1.d0,1.43929d0/)!sum=3.41064
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: whitepoint_e=(/1.d0,1.d0,1.d0/)!sum=3.
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: srgb_red=(/0.64d0,0.33d0,0.03d0/)
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: srgb_grn=(/0.30d0,0.60d0,0.10d0/)
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: srgb_blu=(/0.15d0,0.06d0,0.79d0/)
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: srgb_wht=(/0.95047d0,1.00000d0,1.08883d0/)/3.03930d0
!Current matrices for XYZ to sRGB and backward (from getrgb)
!  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: xyz2srgb_fwd=RESHAPE((/3.240454162114D0,-0.969266030505D0, 0.055643430959D0,&
!                                                               -1.537138512798D0, 1.876010845447D0,-0.204025913517D0,&
!                                                               -0.498531409556D0, 0.041556017530D0, 1.057225188223D0/),(/3,3/))
!  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: xyz2srgb_bwd=RESHAPE((/0.412456439090D0, 0.212672851406D0, 0.019333895582D0,&
!                                                                0.357576077644D0, 0.715152155288D0, 0.119192025881D0,&
!                                                                0.180437483266D0, 0.072174993307D0, 0.950304078536D0/),(/3,3/))
!Old matrix
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: srgb2xyz=RESHAPE((/0.4124d0, 0.3576d0, 0.1805d0, &
                                              0.2126d0, 0.7152d0, 0.0722d0,&
                                              0.0193d0, 0.1192d0, 0.9505d0/),(/3,3/))
! from http://www.brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: srgb3xyz=RESHAPE((/0.4124564d0, 0.3575761d0, 0.1804375D0,&
                                                        0.2126729d0, 0.7151522d0, 0.0721750D0,&
                                                        0.0193339d0, 0.1191920d0, 0.9503041D0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: widergb2xyz=RESHAPE((/0.7161046d0, 0.1009296d0, 0.1471858d0,&
                                                           0.2581874d0, 0.7249378d0, 0.0168748d0,&
                                                           0.0000000d0, 0.0517813d0, 0.7734287d0/),(/3,3/))
! from http://en.wikipedia.org/wiki/LMS_color_space (02.09.2011)
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: srgbwp=RESHAPE((/3.2406d0,-1.5372d0,-0.4986d0,&
                                                   -0.9689d0, 1.8758d0, 0.0415d0,&
                                                    0.0557d0,-0.2040d0, 1.0570d0/),(/3,3/))
! from http://www.brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: srgb3=RESHAPE((/ 3.2404542d0,-1.5371385d0,-0.4985314d0,&
                                                     -0.9692660d0, 1.8760108d0, 0.0415560d0,&
                                                      0.0556434d0,-0.2040259d0, 1.0572252d0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: widergb=RESHAPE((/ 1.4628067d0,-0.1840623d0,-0.2743606d0,&
                                                       -0.5217933d0, 1.4472381d0, 0.0677227d0,&
                                                        0.0349342d0,-0.0968930d0, 1.2884099d0/),(/3,3/))
! from http://www.w3.org/Graphics/Color/sRGB
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: w3srgb=RESHAPE((/3.2410d0,-1.5374d0,-0.4986d0,&
                                                   -0.9692d0, 1.8760d0, 0.0416d0,&
                                                    0.0556d0,-0.2040d0, 1.0570d0/),(/3,3/))
! from own evaluation based on D65 white point XYZ = 0.95047, 1.00000, 1.08883
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: xrgb,xrgb2xyz!dynamic evaluation
!---- Chromatic adation cone response domains (www.brucelindbloom.com)
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: ma_bradford=RESHAPE((/0.8951000d0, 0.2664000d0,-0.1614000d0,&
                                                          -0.7502000d0, 1.7135000d0, 0.0367000d0,&
                                                           0.0389000d0,-0.0685000d0, 1.0296000d0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: ma_vonkries=RESHAPE((/0.4002400d0, 0.7076000d0,-0.0808100d0,&
                                                          -0.2263000d0, 1.1653200d0, 0.0457000d0,&
                                                           0.0000000d0, 0.0000000d0, 0.9182200d0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mainv_bradford,mainv_vonkries

  CONTAINS
    
    SUBROUTINE mapto01(rgb,rgb01,bordermode,crange)
! Trim linear RGB to within [0,1]
! floormode   = left digit
! ceilmode    = right digit (i. bordermode=10 means floormode=1, ceilmode=0)
!            <0 just truncate outside 0 and 1
! floor:      0 scale to c0 = 0
!             1 scale to c0 = minval(rgb)
!             2 scale to c0 = min(minval(rgb),crange(0)) (previous default for crange(0)=0)
!             3 scale to c0 = crange(0) (use =0 for color chart)
! ceiling:    0 scale to c1 = 1
!             1 scale to c1 = maxval(rgb) (the previous default)
!             2 scale to c1 = max(maxval(rgb),crange(1)) 
!             3 scale to c1 = crange(1)
! bordermode -1 don't scale anything, just clip
!           <-1 don't even clip (not recommended for general use!)
! Example: For vivid colorcharts (at least one channel maximised) use bordermode=31
      IMPLICIT NONE
      INTEGER bordermode,floormode,ceilmode,j
      REAL(KIND=CF_DOUBLE) :: rgb(1:3),rgb01(1:3),crange(0:1),c0,c1,scalrgb
      REAL(KIND=CF_DOUBLE) :: grey,gdev,yerg,drgbmin
      REAL(KIND=CF_DOUBLE) :: srgb(1:3)
      IF (bordermode<0) THEN
         IF (bordermode==-1) THEN
            DO j=1,3
               rgb01(j)=max(min(rgb(j),1.d0),0.d0)!just clip
            ENDDO
         ELSE
            rgb01=rgb
         ENDIF
         RETURN
      ENDIF
      floormode=bordermode/10
      ceilmode=bordermode-10*floormode
      IF (floormode==1) THEN
         c0=minval(rgb)
      ELSE IF (floormode==2) THEN
         c0=min(minval(rgb),crange(0))
      ELSE IF (floormode==3) THEN
         c0=crange(0)
      ELSE
         c0=0.d0
      ENDIF
      IF (ceilmode==1) THEN
!  Full value of at least one channel
         c1=maxval(rgb)
      ELSE IF (ceilmode==2) THEN
!  Allow lower values
         c1=max(maxval(rgb),crange(1))
      ELSE IF (ceilmode==3) THEN
         c1=crange(1)
      ELSE IF (ceilmode==4) THEN
!**** Special mode to maximize grey distance *****
         grey=crange(0)!grey axis
         gdev=max(min(crange(1),1.d0),0.d0)!scaling for saturation
         yerg=1.d0-grey
         DO j=1,3
            IF (rgb(j)>=grey) THEN!distance to ceiling
               srgb(j)=yerg/max(rgb(j)-grey,1.d-12)
            ELSE!distance to floor
               srgb(j)=grey/max(grey-rgb(j),1.d-12)
            ENDIF
         ENDDO
         scalrgb=minval(srgb)*gdev
         rgb01(:)=grey+(scalrgb*(rgb(:)-grey))
!---- Exit here
         RETURN
!*************************************************
      ELSE
         c1=1.d0
      ENDIF
      scalrgb=c1-c0
      DO j=1,3
         rgb01(j)=max(min((rgb(j)-c0)/(c1-c0),1.d0),0.d0)
      ENDDO
    END SUBROUTINE mapto01

    SUBROUTINE desaturate_rgb(rgbin,rgbout,fwhite)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: rgbin(1:3),rgbout(1:3),rgbadd(1:3),fwhite,fcol=0.d0,fcomp
      LOGICAL :: addblack
      addblack=fwhite<0.d0
      IF (addblack) THEN
         fwhite=-fwhite
         rgbadd=(/0.d0,0.d0,0.d0/)
      ELSE
         rgbadd=(/1.d0,1.d0,1.d0/)
      ENDIF
      fcol=1.d0-fwhite
      rgbout(:)=fcol*rgbin(:)+fwhite*rgbadd(:)
      RETURN
    END SUBROUTINE desaturate_rgb

!===============================================================================

  SUBROUTINE wrap_rgb_gamma(Jin,Jout,gamma,direction)
    IMPLICIT NONE
    INTEGER :: direction
    REAL(KIND=CF_DOUBLE) :: gamma
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Jin,Jout
    IF (gamma<1.d-2) THEN!Assume sRGB if gamma is tiny
       CALL srgb_gamma(Jin,Jout,direction)
    ELSE
       CALL corr_gamma(Jin,Jout,gamma,direction)
    ENDIF
    RETURN
  END SUBROUTINE wrap_rgb_gamma

    SUBROUTINE corr_gamma(Jin,Jout,gamma,direction)
      IMPLICIT NONE
      INTEGER :: direction,i
      REAL(KIND=CF_DOUBLE) :: Jin(3),Jout(3),gamma,clin,c
!      write(*,'(a,f6.3,2x,i2)') 'gamma,direction=',gamma,direction
!      write(*,'(a,6(e11.4))')    'Jin            =',Jin
      IF (direction>=1) THEN
! RGB_lin to RGB_gamma
         DO i=1,3
            clin=Jin(i)
            c=sign(abs(clin)**(1.d0/gamma),clin)
!            IF (clin>0.d0) THEN
!               c=sign(abs(clin)**(1.d0/gamma),clin)
!!               write(*,'(a,6(e11.4))') ' >0: clin,c,gamma =',clin,c,gamma
!            ELSE
!!               write(*,'(a,6(e11.4))') '<=0: clin,c,gamma =',clin,c,gamma
!               c=0.d0!handle zero separately
!            ENDIF
            Jout(i)=c
         ENDDO
      ELSE IF (direction<=-1) THEN
! RGB_gamma to RGB_lin
         DO i=1,3
            c=Jin(i)
            clin=sign(abs(c)**gamma,c)
!            IF (c>0.d0) THEN
!               clin=c**gamma
!!               write(*,'(a,6(e11.4))') '-1: clin,c,gamma =',clin,c,gamma
!            ELSE
!               clin=0.d0!handle zero separately
!            ENDIF
            Jout(i)=clin
         ENDDO
      ELSE
! No gamma correction
         Jout=Jin
      ENDIF
!      write(*,'(a,6(e11.4)/)') 'Jout        =',Jout
      RETURN
    END SUBROUTINE corr_gamma

    SUBROUTINE srgb_gamma(Jin,Jout,direction)
      IMPLICIT NONE
      INTEGER :: direction,i
      REAL(KIND=CF_DOUBLE) :: Jin(3),Jout(3)
      REAL(KIND=CF_DOUBLE) :: Jrgblin(3),Jsrgb(3),c,clin
      REAL(KIND=CF_DOUBLE) :: a=5.5d-2
      IF (direction>=1) THEN
! RGB_lin to RGB_gamma
         DO i=1,3
            clin=Jin(i)
            IF (clin<=0.0031308d0) THEN
               c=12.92d0*clin
            ELSE
               c=(1.d0+a)*clin**(1.d0/2.4d0)-a
            ENDIF
            Jout(i)=c
         ENDDO
      ELSE IF (direction<=-1) THEN
! RGB_gamma to RGB_lin
         DO i=1,3
            c=Jin(i)
            IF (c<=0.04045d0) THEN
               clin=c/12.92d0
            ELSE
               clin=((c+a)/(1.d0+a))**2.4d0
            ENDIF
            Jout(i)=clin
         ENDDO
      ELSE
! No gamma correction
         Jout=Jin
      ENDIF
      RETURN
    END SUBROUTINE srgb_gamma

    SUBROUTINE rescale_rgb(rgbi,rgbo,yold,ynew,gamma,mode)
!   NOTE: Rescaling may work incorrectly for SRGB due to non-pure power-law
      IMPLICIT NONE
      INTEGER :: mode
      REAL(KIND=CF_DOUBLE) :: rgbi(3),rgbo(3),yold,ynew,gamma,qy,qintens
      REAL(KIND=CF_DOUBLE) :: rgblin(3),rgbdum1(3),rgbdum2(3)
      IF (mode==0) THEN
         rgbo=rgbi
         RETURN
      ELSE IF (mode<0) THEN!treat yold and ynew as RGB levels
         IF (ynew<=yold) THEN
            qy=ynew/yold
         ELSE
            qy=yold/ynew
         ENDIF
!         qy=ynew/yold
         rgbdum1=(/0.d0,qy,0.d0/)
         CALL wrap_rgb_gamma(rgbdum1,rgbdum2,gamma,-1)
         IF (ynew<=yold) THEN
            qintens=rgbdum2(2)
         ELSE
            qintens=1.d0/rgbdum2(2)
         ENDIF
!         qintens=rgbdum2(2)
      ELSE!assume yold and ynew as Y levels (i.e. linear)
         qintens=ynew/yold
      ENDIF
!         CALL srgb_gamma(rgbi,rgblin,-1)
      CALL wrap_rgb_gamma(rgbi,rgblin,gamma,-1)
      rgblin(:)=rgblin(:)*qintens
!         CALL srgb_gamma(rgblin,rgbo,1)
      CALL wrap_rgb_gamma(rgblin,rgbo,gamma,1)
!      write(*,*)
!      write(*,*) 'yold/ynew =',yold/ynew
!      write(*,*) 'ynew/yold =',ynew/yold
!      write(*,*) 'qy        =',qy
!      write(*,*) 'rgbi =',rgbi
!      write(*,*) 'rgbo =',rgbo
      RETURN
    END SUBROUTINE rescale_rgb

!!!! UNTESTED
    SUBROUTINE whitecorr_rgb(rgbi,rgbo,rgbwhite2new,gamma,mode)
! rgb0/1  : input/output RGB
! rgb2wnew: target whitepoint in input RGB space
      IMPLICIT NONE
      INTEGER :: mode,i
      REAL(KIND=CF_DOUBLE) :: rgbi(3),rgbo(3),rgbwhite2new(3),gamma
      REAL(KIND=CF_DOUBLE) :: rgblin(3),rgbwlin(3),wscal,xtiny=1.d-30
      IF (mode==0) THEN
         rgbo=rgbi
         RETURN
      ELSE
         CALL wrap_rgb_gamma(rgbwhite2new,rgbwlin,gamma,-1)
         IF (mode<0) THEN!avoid RGB(:)>1
!            wscal=max(maxval(rgbwhite2new),1.d0)
            wscal=max(minval(rgbwlin),xtiny)
         ELSE
            wscal=1.d0
         ENDIF
!          write(*,*) 'rgbwhite2new =',rgbwhite2new
         CALL wrap_rgb_gamma(rgbi,rgblin,gamma,-1)
         DO i=1,3
            rgblin(i)=rgblin(i)*wscal/max(rgbwlin(i),xtiny)
         ENDDO
         CALL wrap_rgb_gamma(rgblin,rgbo,gamma,1)
!         if (maxval(rgbo)>1.d0) then
!            write(*,*) 'rgbi =',rgbi
!            write(*,*) 'rgbo =',rgbo
!         endif
      ENDIF
      RETURN
    END SUBROUTINE whitecorr_rgb

    SUBROUTINE whitecorr_d(rgbi,rgbo,CCT,a2rgb,gamma,mode)
! rgb0/1  : input/output RGB
! a2rgb   : Matrix on which input RGB is based on
      IMPLICIT NONE
      INTEGER :: mode,i
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: a2rgb
      REAL(KIND=CF_DOUBLE) :: rgbi(3),rgbo(3),CCT,gamma
      REAL(KIND=CF_DOUBLE) :: xd,yd,mdata(0:2),xyzd(3),rgblin(3),rgbwhite2new(3)
      CALL whitepoint_d(CCT,xd,yd,mdata)
      xyzd=(/xd,yd,1.d0-xd-yd/)
      CALL operate_matrix3x3(a2rgb,xyzd,rgbi)
      CALL mapto01(rgbi,rgblin,21,(/0.d0,1.d0/))
      CALL wrap_rgb_gamma(rgblin,rgbwhite2new,gamma,1)
      CALL whitecorr_rgb(rgbi,rgbo,rgbwhite2new,gamma,mode)
      RETURN
    END SUBROUTINE whitecorr_d
!!!! /UNTESTED

!===============================================================================

  SUBROUTINE xyz2uv35(Jxyz,uv35)
! Judd (1935) uniform chromaticity space
    IMPLICIT NONE
    INTEGER :: i
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Jxyz,xyz
    REAL(KIND=CF_DOUBLE),DIMENSION(2) :: uv35
    REAL(KIND=CF_DOUBLE) :: x,y,u,v,div,sumxyz
    REAL(KIND=CF_DOUBLE) :: a=0.4661d0,b=0.1593d0,c=-0.15735d0,d=0.2424d0,e=0.6581d0
    sumxyz=sum(Jxyz)
    DO i=1,3
       xyz(i)=Jxyz(i)/sumxyz
    ENDDO
    x=xyz(1); y=xyz(2)
    div=y+c*x+d
    u=(a*x+b*y)/div
    v=e*y/div
    uv35=(/u,v/)
    RETURN
  END SUBROUTINE xyz2uv35

  SUBROUTINE uv2xyz35(uv35,xyz)
! Own backward-transformation
    IMPLICIT NONE
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: xyz
    REAL(KIND=CF_DOUBLE),DIMENSION(2) :: uv35
    REAL(KIND=CF_DOUBLE) :: x,y,u,v,umb,vme,cuma,cv,du,dv
    REAL(KIND=CF_DOUBLE) :: a=0.4661d0,b=0.1593d0,c=-0.15735d0,d=0.2424d0,e=0.6581d0
    u=uv35(1); v=uv35(2)
    umb=u-b
    vme=v-e
    cuma=c*u-a
    cv=c*v
    du=d*u
    dv=d*v
    x=(umb*dv-vme*du)/(cuma*vme-cv*umb)
    y=(cuma*dv-cv*du)/(cv*umb-cuma*vme)
    xyz=(/x,y,1.d0-x-y/)
    RETURN
  END SUBROUTINE uv2xyz35

    SUBROUTINE xyz2uv(Jxyz,uv)
! CIE1931 XYZ to CIE 1960 uv
      IMPLICIT NONE
      INTEGER i
      REAL(KIND=CF_DOUBLE) :: Jxyz(3),uv(2),xyz(3),sumxyz,divxy,x,y
      sumxyz=sum(Jxyz)
      DO i=1,3
         xyz(i)=Jxyz(i)/sumxyz
      ENDDO
      x=xyz(1); y=xyz(2)
      divxy=(-2.d0*x+12.d0*y+3.d0)
      uv(1)=4.d0*x/divxy
      uv(2)=6.d0*y/divxy
      RETURN
    END SUBROUTINE xyz2uv

    SUBROUTINE uv2xyz(uv,Jxyz)
! CIE1960 uv to CIE1931 xyz (constant lightness)
      IMPLICIT NONE
      INTEGER i
      REAL(KIND=CF_DOUBLE) :: Jxyz(3),uv(2),xyz(3),sumxyz,divuv,u,v
!      sumxyz=sum(Jxyz)
!      DO i=1,3
!         xyz(i)=Jxyz(i)/sumxyz
!      ENDDO
      u=uv(1); v=uv(2)
      divuv=(2.d0*u-8.d0*v+4.d0)
      xyz(1)=3.d0*u/divuv
      xyz(2)=2.d0*v/divuv
      xyz(3)=1.d0-xyz(1)-xyz(2)
      Jxyz(:)=xyz(:)
      RETURN
    END SUBROUTINE uv2xyz

    SUBROUTINE xyz2uv76(Jxyz,uv76)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: Jxyz(3),uv76(2),uv(2)
      CALL xyz2uv(Jxyz,uv)
      uv76(1)=uv(1)
      uv76(2)=uv(2)*1.5d0
      RETURN
    END SUBROUTINE xyz2uv76

    SUBROUTINE uv2xyz76(uv76,Jxyz)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: Jxyz(3),uv76(2),uv(2)
      uv(1)=uv76(1)
      uv(2)=uv76(2)/1.5d0
      CALL uv2xyz(uv,Jxyz)
      RETURN
    END SUBROUTINE uv2xyz76

    SUBROUTINE uvy2uvw64(uv,y100,uvn,uvw100)
! xyz100 means xyz~100 = 100*Jxyz
! For illuminants use Y=xyz(2)=100.
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: uv(2),y100,uvn(2),uvw100(3)
      REAL(KIND=CF_DOUBLE) :: ustar,vstar,wstar,third=1.d0/3.d0
      wstar=25.d0*y100**third-17.d0
      ustar=13.d0*wstar*(uv(1)-uvn(1))
      vstar=13.d0*wstar*(uv(2)-uvn(2))
      uvw100=(/ustar,vstar,wstar/)
      RETURN
    END SUBROUTINE uvy2uvw64

    SUBROUTINE xyz2uvw64(xyz100,xyzwhite,uvw100)
! xyz100 means xyz~100 = 100*Jxyz
! For illuminants use Y=xyz(2)=100.
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: xyz100(3),xyzwhite(3),uvw100(3)
      REAL(KIND=CF_DOUBLE) :: ustar,vstar,wstar,third=1.d0/3.d0
      REAL(KIND=CF_DOUBLE) :: y100,uv(2),uvn(2),sumxyz,divuv
      y100=xyz100(2)
      CALL xyz2uv(xyz100,uv)
      CALL xyz2uv(xyzwhite,uvn)
      CALL uvy2uvw64(uv,y100,uvn,uvw100)
!      wstar=25.d0*y100**third-17.d0
!      ustar=13.d0*wstar*(uv(1)-uvn(1))
!      vstar=13.d0*wstar*(uv(2)-uvn(2))
!      uvw100=(/ustar,vstar,wstar/)
      RETURN
    END SUBROUTINE xyz2uvw64

    SUBROUTINE xyz2lab(XYZ,XYZwhite,Lab)
! XYZ assumed to be absolute (i.e. Y = luminance)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: Lab(3),XYZ(3),XYZwhite(3),L16
      REAL(KIND=CF_DOUBLE) :: L,a,b,xr,yr,zr,fx,fy,fz
      xr=XYZ(1)/XYZwhite(1)
      yr=XYZ(2)/XYZwhite(2)
      zr=XYZ(3)/XYZwhite(3)
      fx=flab_fwd(xr)
      fy=flab_fwd(yr)
      fz=flab_fwd(zr)
      L=116.d0*fy-16.d0
      a=500.d0*(fx-fy)
      b=200.d0*(fy-fz)
      Lab=(/L,a,b/)
      RETURN
    END SUBROUTINE xyz2lab

    SUBROUTINE lab2xyz(Lab,XYZ,XYZwhite)
! Input:
!     Lab(1:3)    - Vector that contains the L,a,b data
! Output:
!     XYZ(1:3)    - Vector that contains the X,Y,Z data
! Parameter:
!     XYZwhite    - Vector that contains the whitepoint X,Y,Z
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: Lab(3),XYZ(3),XYZwhite(3),L16
      REAL(KIND=CF_DOUBLE) :: L,a,b
      L=Lab(1);a=Lab(2);b=Lab(3)
      L16=(L+16.d0)/116.d0
      XYZ(1)=XYZwhite(1)*flab_bwd(L16+a/500.d0)
      XYZ(2)=XYZwhite(2)*flab_bwd(L16)
      XYZ(3)=XYZwhite(3)*flab_bwd(L16-b/200.d0)!sign corrected here
!      XYZ(3)=XYZwhite(3)*flab_bwd(L16+b/200.d0)!old wrong formula
!      write(*,*) 'x,y,z =',xyz(1),xyz(2),xyz(3)
      RETURN
    END SUBROUTINE lab2xyz

    FUNCTION flab_fwd(t)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: flab_fwd,t
      REAL(KIND=CF_DOUBLE) :: c1=6.d0/29.d0,c2=4.d0/29.d0,third=1.d0/3.d0
      IF (t>c1**3) THEN
         flab_fwd=t**third
      ELSE
         flab_fwd=t/(3.d0*c1**2)+c2
      ENDIF
    END FUNCTION flab_fwd

    FUNCTION flab_bwd(t)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: flab_bwd,t
      REAL(KIND=CF_DOUBLE) :: c1=6.d0/29.d0,c2=4.d0/29.d0
      IF (t>c1) THEN
         flab_bwd=t**3
      ELSE
         flab_bwd=3.d0*c1**2*(t-c2)
      ENDIF
    END FUNCTION flab_bwd


!===============================================================================

    SUBROUTINE whitepoint_d(cct,xd,yd,mdata)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: cct,xd,yd,mdata(0:2)
      REAL(KIND=CF_DOUBLE) :: a0=0.244063D0,a1=0.09911D0,a2=2.9678D0,a3=-4.6070D0
      REAL(KIND=CF_DOUBLE) :: b0=0.237040D0,b1=0.24748D0,b2=1.9018D0,b3=-2.0064D0
      REAL(KIND=CF_DOUBLE) :: c0=-0.275D0,c1=2.870D0,c2=-3.000D0
      REAL(KIND=CF_DOUBLE) :: s
!==== Compute chromaticity
      s=1.d3/cct
      IF (cct.le.7.d3) THEN
         xd=a0+(a1+(a2+a3*s)*s)*s
      ELSE
         xd=b0+(b1+(b2+b3*s)*s)*s
      ENDIF
      yd=c0+(c1+c2*xd)*xd
!==== Coefficients
      mdata(0)=0.0241D0+0.2562D0*xd-0.7341D0*yd
      mdata(1)=(-1.3515D0-1.7703D0*xd+5.9114D0*yd)/mdata(0)
      mdata(2)=(0.03000D0-31.4424D0*xd+30.0717D0*yd)/mdata(0)
      RETURN
    END SUBROUTINE whitepoint_d

    SUBROUTINE brightness_correction(Jxyz,f,kf)
!==== Correction factor for brightness for constand luminance
!     as a function of chromaticity
!     (Kaiser 1986, retrieved from Ohta & Robertsen,
!     Colorimetry - Fundamentals and Applications,
!     2005, equation 1.40)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: Jxyz(3),x,y,f,kf,scalxyz
      scalxyz=sum(Jxyz)
      x=Jxyz(1)/scalxyz; y=Jxyz(2)/scalxyz
      f=0.256d0-0.184d0*y-2.527d0*x*y+4.656d0*x**3*y+4.657d0*x*y**4
      kf=1.d1**f
      RETURN
    END SUBROUTINE brightness_correction

!===============================================================================

    SUBROUTINE operate_matrix3x3(mat,vec0,vec1)
!     vec1 = mat vec0
!     mat is a 3x3 matrix of the following form:
!     /a11 a12 a13\   /mat(1,1) mat(2,1) mat(3,1)\
!     |a21 a22 a23| = |mat(1,2) mat(2,2) mat(3,2)|
!     \a31 a32 a33/   \mat(1,3) mat(2,3) mat(3,3)/
!
!     Note that mat(j,i) = aij (left entry revolves fastest)
      IMPLICIT NONE
      INTEGER i,j
      REAL(KIND=CF_DOUBLE) mat(3,3),vec0(3),vec1(3)
      vec1(1)=0.d0
      vec1(2)=0.d0
      vec1(3)=0.d0
      DO i=1,3
         DO j=1,3
            vec1(i)=vec1(i)+mat(j,i)*vec0(j)!inverse order of i,j for optimum memory acces
         ENDDO
      ENDDO
      RETURN
    END SUBROUTINE operate_matrix3x3

    SUBROUTINE invert_matrix3x3(mat,tam,det)
      IMPLICIT NONE
      INTEGER i,j,ii
      REAL(KIND=CF_DOUBLE) mat(3,3),tam(3,3),det
      det=0.d0
      det=  mat(1,1)*mat(2,2)*mat(3,3) - mat(3,1)*mat(2,2)*mat(1,3)  &
           + mat(2,1)*mat(3,2)*mat(1,3) - mat(2,1)*mat(1,2)*mat(3,3) &
           + mat(3,1)*mat(1,2)*mat(2,3) - mat(1,1)*mat(3,2)*mat(2,3)
!     (a  b  c)               (ei-fh  ch-bi  bf-ce)
!     (d  e  f) inv = 1/det * (fg-di  ai-cg  cd-af)
!     (g  h  i)               (dh-eg  bg-ah  ae-bd)
!
!     (a11  b12  c13)               (e22*i33-f23*h32  c13*h32-b12*i33  b12*f23-c13*e22)
!     (d21  e22  f23) inv = 1/det * (f23*g31-d21*i33  a11*i33-c13*g31  c13*d21-a11*f23)
!     (g31  h32  i33)               (d21*h32-e22*g31  b12*g31-a11*h32  a11*e22-b12*d21)
!
!     (a11  a12  a13)               (a22*a33-a23*h32  a13*a32-a12*i33  a12*a23-a13*a22)
!     (a21  a22  a23) inv = 1/det * (a23*a31-a21*i33  a11*a33-a13*g31  a13*a21-a11*a23)
!     (a31  a32  a33)               (a21*a32-a22*g31  a12*a31-a11*h32  a11*a22-a12*a21)
      tam(1,1)=mat(2,2)*mat(3,3)-mat(3,2)*mat(2,3)
      tam(2,1)=mat(3,1)*mat(2,3)-mat(2,1)*mat(3,3)
      tam(3,1)=mat(2,1)*mat(3,2)-mat(3,1)*mat(2,2)
      
      tam(1,2)=mat(3,2)*mat(1,3)-mat(1,2)*mat(3,3)
      tam(2,2)=mat(1,1)*mat(3,3)-mat(3,1)*mat(1,3)
      tam(3,2)=mat(3,1)*mat(1,2)-mat(1,1)*mat(3,2)
      
      tam(1,3)=mat(1,2)*mat(2,3)-mat(2,2)*mat(1,3)
      tam(2,3)=mat(2,1)*mat(1,3)-mat(1,1)*mat(2,3)
      tam(3,3)=mat(1,1)*mat(2,2)-mat(2,1)*mat(1,2)
      DO i=1,3
         DO j=1,3
            tam(j,i)=tam(j,i)/det
         ENDDO
      ENDDO
      RETURN
    END SUBROUTINE invert_matrix3x3

    SUBROUTINE product_matrix3x3(mat1,mat2,mat12)
      IMPLICIT NONE
      INTEGER i,j,k
      REAL(KIND=CF_DOUBLE) mat1(3,3),mat2(3,3),mat12(3,3),sum
      DO i=1,3
         DO j=1,3
            sum=0.d0
            DO k=1,3
               sum=sum+mat1(k,i)*mat2(j,k)
            ENDDO
            mat12(j,i)=sum
         ENDDO
      ENDDO
      RETURN
    END SUBROUTINE product_matrix3x3
          
    SUBROUTINE transpose_matrix3x3(a,atp)
      IMPLICIT NONE
      INTEGER i,j
      REAL(KIND=CF_DOUBLE) a(3,3),atp(3,3)
      DO i=1,3
         DO j=1,3
            atp(j,i)=a(i,j)
         ENDDO
      ENDDO
      RETURN
    END SUBROUTINE transpose_matrix3x3
   
!===============================================================================

    SUBROUTINE getrgb(xyzr,xyzg,xyzb,xyzw,axyz2rgb,argb2xyz)
! XYZw should be scaled to Y=1 for standard matrix
      IMPLICIT NONE
      INTEGER j
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: xyzr,xyzg,xyzb,xyzw
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: testxyzr,testxyzg,testxyzb,testxyzw
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: testr,testg,testb,testw
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: oner,oneg,oneb,onew
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: axyz2rgb,argb2xyz
      REAL(KIND=CF_DOUBLE) :: k,c1,c2,c3,adet
      REAL(KIND=CF_DOUBLE) :: a11,a12,a13, a21,a22,a23, a31,a32,a33
!    Set R=0
      k=xyzg(1)/xyzb(1)
!     WRITE(*,*) 'kr1 =',k
      c1=0.d0
      c2=xyzg(2)-k*xyzb(2)
      c3=xyzg(3)-k*xyzb(3)
      a12=-c3/c2
      k=xyzg(2)/xyzb(2)
!     WRITE(*,*) 'kr2 =',k
      c1=xyzg(1)-k*xyzb(1)
      c2=0.d0
      c3=xyzg(3)-k*xyzb(3)
      a11=-c3/c1
!    Set G=0
      k=xyzr(1)/xyzb(1)
!     WRITE(*,*) 'kg1 =',k
      c1=0.d0
      c2=xyzr(2)-k*xyzb(2)
      c3=xyzr(3)-k*xyzb(3)
      a22=-c3/c2
      k=xyzr(2)/xyzb(2)
!     WRITE(*,*) 'kg2 =',k
      c1=xyzr(1)-k*xyzb(1)
      c2=0.d0
      c3=xyzr(3)-k*xyzb(3)
      a21=-c3/c1
!    Set B=0
      k=xyzr(1)/xyzg(1)
!     WRITE(*,*) 'kb1 =',k
      c1=0.d0
      c2=xyzr(2)-k*xyzg(2)
      c3=xyzr(3)-k*xyzg(3)
      a32=-c3/c2
      k=xyzr(2)/xyzg(2)
!     WRITE(*,*) 'kb2 =',k
      c1=xyzr(1)-k*xyzg(1)
      c2=0.d0
      c3=xyzr(3)-k*xyzg(3)
      a31=-c3/c1
!
!
      a13=1.d0/(a11*xyzw(1)+a12*xyzw(2)+xyzw(3))
      a23=1.d0/(a21*xyzw(1)+a22*xyzw(2)+xyzw(3))
      a33=1.d0/(a31*xyzw(1)+a32*xyzw(2)+xyzw(3))
      a11=a11*a13; a12=a12*a13
      a21=a21*a23; a22=a22*a23
      a31=a31*a33; a32=a32*a33
      axyz2rgb=RESHAPE((/a11,a12,a13,&
           a21,a22,a23,&
           a31,a32,a33/),(/3,3/))
      CALL invert_matrix3x3(axyz2rgb,argb2xyz,adet)
      IF (cf_verbose<1) RETURN!skip test output
      WRITE(*,*)
      WRITE(*,'(A)') '************************************************************'
      WRITE(*,'(A)') 'GETRGB: Show Matrices'
      WRITE(*,'(A)') '        XYZ to RGB'
      WRITE(*,'(A,3(1X,F16.12))') 'a11,a12,a13 =',(axyz2rgb(1,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'a21,a22,a23 =',(axyz2rgb(2,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'a31,a32,a33 =',(axyz2rgb(3,j),j=1,3)
      WRITE(*,*)
      WRITE(*,'(A)') '        RGB to XYZ'
      WRITE(*,'(A,3(1X,F16.12))') 'b11,b12,b13 =',(argb2xyz(1,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'b21,b22,b23 =',(argb2xyz(2,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'b31,b32,b33 =',(argb2xyz(3,j),j=1,3)
      WRITE(*,*)
      WRITE(*,'(A)') 'GETRGB: Test Matrices'
      WRITE(*,'(A)') '        XYZ to RGB'
      testxyzr=xyzr;testxyzg=xyzg;testxyzb=xyzb;testxyzw=xyzw
      CALL operate_matrix3x3(axyz2rgb,testxyzr,testr)
      CALL operate_matrix3x3(axyz2rgb,testxyzg,testg)
      CALL operate_matrix3x3(axyz2rgb,testxyzb,testb)
      CALL operate_matrix3x3(axyz2rgb,testxyzw,testw)
      WRITE(*,'(A,3(1X,F16.12))') 'RGB(R) =',(testr(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'RGB(G) =',(testg(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'RGB(B) =',(testb(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'RGB(W) =',(testw(j),j=1,3)
      CALL operate_matrix3x3(argb2xyz,testr,testxyzr)
      CALL operate_matrix3x3(argb2xyz,testg,testxyzg)
      CALL operate_matrix3x3(argb2xyz,testb,testxyzb)
      CALL operate_matrix3x3(argb2xyz,testw,testxyzw)
      WRITE(*,*)
      WRITE(*,'(A)') '        RGB to XYZ'
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(R) =',(testxyzr(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(G) =',(testxyzg(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(B) =',(testxyzb(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(W) =',(testxyzw(j),j=1,3)
      oner=(/1.d0,0.d0,0.d0/)
      oneg=(/0.d0,1.d0,0.d0/)
      oneb=(/0.d0,0.d0,1.d0/)
      onew=(/1.d0,1.d0,1.d0/)
      CALL operate_matrix3x3(argb2xyz,oner,testxyzr)
      CALL operate_matrix3x3(argb2xyz,oneg,testxyzg)
      CALL operate_matrix3x3(argb2xyz,oneb,testxyzb)
      CALL operate_matrix3x3(argb2xyz,onew,testxyzw)
      WRITE(*,*)
      WRITE(*,'(A)') '        RGB uni to XYZ'
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(R) =',(testxyzr(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(G) =',(testxyzg(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(B) =',(testxyzb(j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'xyz(W) =',(testxyzw(j),j=1,3)
      WRITE(*,'(A)') '************************************************************'
      WRITE(*,*)
      RETURN
    END SUBROUTINE getrgb

    SUBROUTINE XYZ2srgb(Jxyz,ymode,rgblin,srgb,a2rgb,bordermode,crange)
!     ylum = 0: no luminance scaling
!            1: assume Ymax = 1
!            2: assume Ymax = 100
      IMPLICIT NONE
      INTEGER :: ymode,bordermode
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: a2rgb
      REAL(KIND=CF_DOUBLE),DIMENSION(2) :: crange
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Jxyz,xyz,XYZwhite,rgb0,rgblin,srgb0,srgb
      REAL(KIND=CF_DOUBLE) :: scalxyz,ylum!,x,y,z,xw,yw
      scalxyz=sum(Jxyz)
      IF (abs(ymode)==1) THEN
         ylum=Jxyz(2)
         xyz(:)=Jxyz(:)
!         write(*,*) 'ymode=1: ylum =',ylum
      ELSE IF (ymode==2) THEN
         ylum=Jxyz(2)/1.d2
         xyz(:)=Jxyz(:)/1.d2
!         write(*,*) 'ymode=2: ylum =',ylum
      ELSE
         ylum=1.d0
         xyz(:)=Jxyz(:)/scalxyz
!         write(*,*) 'ymode=0'
      ENDIF
!! Whitepoint
!      IF (xw*yw<=0.d0) THEN
!         XYZwhite(:)=whitepoint_d65(:)
!      ELSE
!         XYZwhite(1)=xw
!         XYZwhite(2)=yw
!         XYZwhite(3)=1.d0-xw-yw
!      ENDIF
      CALL operate_matrix3x3(a2rgb,xyz,rgb0)
      IF (ymode>=0) THEN
!         write(*,*) 'ymode =',ymode
         CALL mapto01(rgb0,rgblin,bordermode,crange)
      ELSE
         rgblin=rgb0
      ENDIF
!      IF (ymode>0) THEN
!         rgb0(:)=rgb0(:)*ylum
!      ENDIF
      CALL srgb_gamma(rgblin,srgb,1)
!      CALL mapto01(rgb0,rgblin,bordermode,crange)
!      CALL mapto01(srgb0,srgb,bordermode,crange)
      RETURN
    END SUBROUTINE XYZ2srgb

    FUNCTION cf_safepower(x,pwr)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE) :: cf_safepower,x,pwr
      cf_safepower=sign(abs(x)**pwr,x)!absolute value of x to pwr times sign of x
      RETURN
    END FUNCTION cf_safepower

  END MODULE color_functions

!===============================================================================
!==== Chromatic adaptation functions ===========================================
!===============================================================================

! Note: Matrix-based chromatic adaptation may fail for combinations of extreme
!       whitepoints (like illuminant A) and near-monochromatic test colors
!       (negative Y values after applying CA).
!       Therefore, any colour space using such CA may run amok (i.e. produce NaNs)
!       for extreme whitepoints & colours.

  MODULE chromadapt_matrices
    USE color_functions
!    INTEGER,PARAMETER :: CF_SINGLE=4
!    INTEGER,PARAMETER :: CF_DOUBLE=8
!   data from www.brucelindbloom.com
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: vonkries_fwd=RESHAPE((/0.4002400d0, 0.7076000d0,-0.0808100d0, &
                                                                 -0.2263000d0, 1.1653200d0, 0.0457000d0, &
                                                                  0.0000000d0, 0.0000000d0, 0.9182200d0/),(/3,3/))
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: bradford_fwd=RESHAPE((/0.8951000d0, 0.2664000d0,-0.1614000d0, &
                                                                 -0.7502000d0, 1.7135000d0, 0.0367000d0, &
                                                                  0.0389000d0,-0.0685000d0, 1.0296000d0/),(/3,3/))
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: ciecat02_fwd=RESHAPE((/0.7328d0, 0.4296d0,-0.1624d0, &
                                                                 -0.7036d0, 1.6975d0, 0.0061d0, &
                                                                  0.0030d0, 0.0136d0, 0.9834d0/),(/3,3/))
!Modified CAT02 after Brill and Süsstrunk
!See also description of http://commons.wikimedia.org/wiki/File:Linear_visible_spectrum.svg
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: modcatbs_fwd=RESHAPE((/0.7328d0, 0.4296d0,-0.1624d0, &
                                                                 -0.7036d0, 1.6975d0, 0.0061d0, &
                                                                  0.0000d0, 0.0000d0, 1.0000d0/),(/3,3/))
!Modified CAT02 by Li et al. (2009, submitted to Color Res.Appl [accepted?]),
!See http://rua.ua.es/dspace/bitstream/10045/18684/1/CAT02-CRA.pdf
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: modcatli_fwd=RESHAPE((/1.007245d0, 0.011136d0,-0.018381d0, &
                                                                 -0.318061d0, 1.314589d0, 0.003471d0, &
                                                                  0.000000d0, 0.000000d0, 1.000000d0/),(/3,3/))
!Hunter-Pointer-Estévez-Space conversion matrix
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: hpespace_fwd=RESHAPE((/0.38971d0, 0.68898d0,-0.07868d0, &
                                                                 -0.22981d0, 1.18340d0, 0.04641d0, &
                                                                  0.00000d0, 0.00000d0, 1.00000d0/),(/3,3/))
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: vonkries_bwd,bradford_bwd,ciecat02_bwd,hpespace_bwd
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: modcatbs_bwd,modcatli_bwd
    REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mcat_fwd,mcat_bwd
    REAL(KIND=CF_DOUBLE) :: det_vonkries,det_bradford,det_ciecat02,det_modcatbs,det_modcatli,det_hpespace,det_mcat
    INTEGER :: flg_catmod=0
    LOGICAL :: got_inverse_matrices=.FALSE.,got_mcat=.FALSE.
!Auxiliary flags (e.g. for individual inverse matrices)
    LOGICAL :: got_inv00=.FALSE.,got_inv01=.FALSE.,got_inv02=.FALSE.,got_inv03=.FALSE.

  CONTAINS

    SUBROUTINE chromadapt_get_inverse
      IMPLICIT NONE
      DOUBLE PRECISION :: det
      CALL invert_matrix3x3(vonkries_fwd,vonkries_bwd,det_vonkries)
      CALL invert_matrix3x3(bradford_fwd,bradford_bwd,det_bradford)
      CALL invert_matrix3x3(ciecat02_fwd,ciecat02_bwd,det_ciecat02)
      CALL invert_matrix3x3(modcatbs_fwd,modcatbs_bwd,det_modcatbs)
      CALL invert_matrix3x3(modcatli_fwd,modcatli_bwd,det_modcatli)
      got_inverse_matrices=.TRUE.
      RETURN
    END SUBROUTINE chromadapt_get_inverse

    SUBROUTINE chromadapt_xyz(Ixyz_s,Ixyz_ws,Ixyz_wd, Ixyz_d, selmatrix)
      IMPLICIT NONE
      INTEGER :: selmatrix,j
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Ixyz_s,Ixyz_ws,Ixyz_wd,Ixyz_d
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: rhos,rhod
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat,tam,mcrho
      REAL(KIND=CF_DOUBLE) :: rs,gs,bs,rd,gd,bd
      IF (.not.got_inverse_matrices) THEN
!         write(*,*) '>>>>>>>> selmatrix =',selmatrix
         CALL chromadapt_get_inverse
      ENDIF
!---- Select transformation matrices
      IF (selmatrix==1) THEN
         mat=vonkries_fwd
         tam=vonkries_bwd
      ELSE IF (selmatrix==2) THEN
         mat=bradford_fwd
         tam=bradford_bwd
      ELSE IF (selmatrix==3) THEN
         mat=ciecat02_fwd
         tam=ciecat02_bwd
      ELSE IF (selmatrix==4) THEN
         mat=modcatbs_fwd
         tam=modcatbs_bwd
      ELSE IF (selmatrix==5) THEN
         mat=modcatli_fwd
         tam=modcatli_bwd
      ELSE
         mat=RESHAPE((/1.d0,0.d0,0.d0, 0.d0,1.d0,0.d0, 0.d0,0.d0,1.d0/),(/3,3/))
         tam=mat
      ENDIF
!---- Get illuminant rho gamma beta ratios (contained in the rho vectors)
      CALL operate_matrix3x3(mat,Ixyz_ws,rhos)
      CALL operate_matrix3x3(mat,Ixyz_wd,rhod)
      rs=rhos(1); gs=rhos(2); bs=rhos(3)
      rd=rhod(1); gd=rhod(2); bd=rhod(3)
!      write(*,*) 'Ixyz_ws,Ixyz_wd =',Ixyz_ws,'; ',Ixyz_wd
!      write(*,*) 'rhos,rhod = ',rhos,'; ',rhod
!---- Get chromadapt matrix
      mcrho=RESHAPE((/rd/rs,0.d0,0.d0, 0.d0,gd/gs,0.d0, 0.d0,0.d0,bd/bs/),(/3,3/))
!      CALL product_matrix3x3(mat1,mat2,mat12)
!---- Apply to object color
      CALL operate_matrix3x3(mat,Ixyz_s,rhos)!re-use rhos here
      CALL operate_matrix3x3(mcrho,rhos,rhod)
      CALL operate_matrix3x3(tam,rhod,Ixyz_d)
      RETURN
! DEBUGGING CENTER
!      write(*,*) 'Ixyz_s,Ixyz_d '
      WRITE(*,'(A,3(1X,F16.12))') 'mat11,mat12,mat13 =',(mat(1,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'mat21,mat22,mat23 =',(mat(2,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'mat31,mat32,mat33 =',(mat(3,j),j=1,3)

      WRITE(*,'(A,3(1X,F16.12))') 'tam11,tam12,tam13 =',(tam(1,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'tam21,tam22,tam23 =',(tam(2,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'tam31,tam32,tam33 =',(tam(3,j),j=1,3)

      WRITE(*,'(A,3(1X,F16.12))') 'mcrho11,mcrho12,mcrho13 =',(mcrho(1,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'mcrho21,mcrho22,mcrho23 =',(mcrho(2,j),j=1,3)
      WRITE(*,'(A,3(1X,F16.12))') 'mcrho31,mcrho32,mcrho33 =',(mat(3,j),j=1,3)
      stop
    END SUBROUTINE chromadapt_xyz
    
    SUBROUTINE chromadapt_uv60(uv_s,uv_ws,uv_wd, uv_d, selmatrix)
      IMPLICIT NONE
      INTEGER :: selmatrix
      REAL(KIND=CF_DOUBLE),DIMENSION(2) :: uv_s,uv_ws,uv_wd, uv_d
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Ixyz_s,Ixyz_ws,Ixyz_wd,Ixyz_d
!      write(*,*) 'uv_ws,uv_wd =',uv_ws,';   ',uv_wd
      CALL uv2xyz(uv_s,Ixyz_s)
      CALL uv2xyz(uv_ws,Ixyz_ws)
      CALL uv2xyz(uv_wd,Ixyz_wd)
      CALL chromadapt_xyz(Ixyz_s,Ixyz_ws,Ixyz_wd, Ixyz_d, selmatrix)
!      Ixyz_d=Ixyz_s
!      write(*,*) 'Ixyz_s,Ixyz_d =',Ixyz_s,';   ',Ixyz_d
!      stop
      CALL xyz2uv(Ixyz_d,uv_d)
    END SUBROUTINE chromadapt_uv60
  END MODULE chromadapt_matrices

!===============================================================================
!==== CIECAM02 and related functions ===========================================
!===============================================================================

MODULE color_functions_ciecam02
  USE color_functions_mathconst
  USE color_functions
  USE chromadapt_matrices
  IMPLICIT NONE
! Some global variables for optional wrapping
  INTEGER :: surround_condition_select=1
  REAL(KIND=CF_DOUBLE) :: cdm2_illum=500.d0!illumination in cd/m^2
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_background=(/20.d0,20.d0,20.d0/)! Average grey 20%
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_illuminant=(/100.d0,100.d0,100.d0/)! Equal energy illuminant
  REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_reference=(/100.d0,100.d0,100.d0/)! same as reference
  REAL(KIND=CF_DOUBLE),DIMENSION(3,5) :: hq_table=RESHAPE((/ 20.14d0,0.8d0,  0.d0, &
                                                             90.00d0,0.7d0,100.d0, &
                                                            164.25d0,1.0d0,200.d0, &
                                                            237.53d0,1.2d0,300.d0, &
                                                            380.14d0,0.8d0,400.d0/),(/3,5/))
  
CONTAINS

  SUBROUTINE wrap_ciecam02(IXYZio,JChio,QMHio,qmode,inpmode,direction)
! works like fwd and bwd, but environment parameters are transferred via module
    IMPLICIT NONE
    INTEGER :: direction,qmode,inpmode,flgxyzunity,degmode,surround_condition
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZio,JChio,QMhio,JChQMH
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ,IXYZb,IXYZw,IXYZwr,JCh,QMh
    REAL(KIND=CF_DOUBLE) :: Ew,Lw
    flgxyzunity=inpmode/10
    degmode=inpmode-10*flgxyzunity
    surround_condition=surround_condition_select
    Lw=cdm2_illum
    IXYZb=IXYZ_background
    IXYZw=IXYZ_illuminant
    IXYZwr=IXYZ_reference
    IF (cf_verbose>=1) THEN
       WRITE(*,*) 
       WRITE(*,*) 'qmode  =',qmode
       WRITE(*,*) 'JChio  =',JChio
       WRITE(*,*) 'QMHio  =',QMHio
       WRITE(*,*) 'JChQMH =',JChQMH
       WRITE(*,*) 
    ENDIF
    IF (direction>=1) THEN
       IF (flgxyzunity<=0) THEN
          IXYZ=IXYZio
       ELSE
          IXYZ=1.d2*IXYZio
       ENDIF
! Call ciecam02_fwd with default options, i.e. inpmode=01 (unity, degrees)
       CALL ciecam02_fwd(IXYZ,IXYZb,IXYZw,IXYZwr, &
            surround_condition,Lw,JCh,QMH,1)
       IF (degmode>=1) THEN!ouput in degrees -> just copy values
          JChio=(/JCh(1),JCh(2),JCh(3)/)
       ELSE!output in radians -> change h
          JChio=(/JCh(1),JCh(2),JCh(3)*d2r/)
       ENDIF
       QMHio=QMH
    ELSE IF (direction<=-1) THEN
       IF (qmode<=0) THEN
          IF (degmode>=1) THEN!input in degrees -> just copy values
             JCh=(/JChio(1),JChio(2),JChio(3)/)
          ELSE!input in radians -> change h
             JCh=(/JChio(1),JChio(2),JChio(3)*r2d/)
          ENDIF
          JChQMH=JCh
          IF (cf_verbose>=1) WRITE(*,*) 'JChio   =',JChio
          IF (cf_verbose>=1) WRITE(*,*) 'JCh     =',JCh
          IF (cf_verbose>=1) WRITE(*,*) 'JChQMH  =',JChQMH
       ELSE IF (qmode==1) THEN
!          JChQMH=(/JChio(1),JChio(2),QMHio(3)/)!mixed input; non-intuitive
          JChQMH=JChio!take JCh but silently assume that the user has assigned JCh(3) to Hquad
       ELSE
          JChQMH=QMHio
       ENDIF
       CALL ciecam02_bwd(JChQMH,qmode,IXYZ,IXYZb,IXYZw,IXYZwr, &
            surround_condition,Lw,1)
       IF (flgxyzunity<=0) THEN!just copy
          IXYZio=IXYZ
       ELSE!divide by 100 to unity
          IXYZio=1.d-2*IXYZ
          IF (cf_verbose>=2) WRITE(*,*) 'IXYZ(2),IXYZio(2) =',IXYZ(2),IXYZio(2)
       ENDIF
    ENDIF
!    WRITE(*,*) '',
    RETURN
  END SUBROUTINE wrap_ciecam02

  SUBROUTINE wrap_ciecam02_xyzdiff(IXYZi1,IXYZi2,DE02,c02mode,flgxyzunity)
    IMPLICIT NONE
    INTEGER :: inpmode,c02mode,flgxyzunity,surround_condition
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZi1,IXYZi2,JChio,QMhio
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ1,IXYZ2,IXYZb,IXYZw,IXYZwr,JCh,QMh
    REAL(KIND=CF_DOUBLE) :: Lw,DE02
    inpmode=10*flgxyzunity+1!assume degmode==1 as default
    surround_condition=surround_condition_select
    Lw=cdm2_illum
    IXYZb=IXYZ_background
    IXYZw=IXYZ_illuminant
    IXYZwr=IXYZ_reference
    IF (flgxyzunity<=0) THEN
       IXYZ1=IXYZi1; IXYZ2=IXYZi2
    ELSE
       IXYZ1=1.d2*IXYZi1; IXYZ2=1.d2*IXYZi2
    ENDIF
    CALL ciecam02_xyzdiff(IXYZi1,IXYZi2,IXYZb,IXYZw,IXYZwr, &
         surround_condition,Lw,DE02,c02mode,inpmode)
    RETURN
  END SUBROUTINE wrap_ciecam02_xyzdiff

!-- CIECAM02 retrieved from Wikipedia, J. Schanda:Colorimetry and thesis by Yang Zue (2008, inverse and Euclidean)
  SUBROUTINE ciecam02_precalcs(surround_condition,Lw,IXYZb,IXYZw,IXYZwr,LMSw,LMSwr, &
       Yb,Yw,nbw,Nbb,z,F,FL,LA,D,c,Nc,k)
    IMPLICIT NONE
    INTEGER :: surround_condition!0:dark,1:dim,2:average
    INTEGER :: i,j
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZb,IXYZw,IXYZwr,LMSw,LMSwr
    REAL(KIND=CF_DOUBLE) :: k,F,D,c,Nc,FL,La,Lw,Yb,Yw,nbw,Nbb,z
!    LOGICAL :: setup_cambwd=.TRUE.
!    SAVE setup_cambwd
    IF (.NOT.got_mcat) THEN
       IF (flg_catmod==1) THEN
          mcat_fwd=modcatbs_fwd
       ELSE IF (flg_catmod==2) THEN
          mcat_fwd=modcatli_fwd
       ELSE
          mcat_fwd=ciecat02_fwd
       ENDIF
       CALL invert_matrix3x3(mcat_fwd,mcat_bwd,det_mcat)
       CALL invert_matrix3x3(hpespace_fwd,hpespace_bwd,det_hpespace)
       got_mcat=.TRUE.
    ENDIF
    IF (surround_condition<=0) THEN
       F=0.8d0
       c=0.525d0
       Nc=0.8d0
    ELSE IF (surround_condition==1) THEN
       F=0.9d0
       c=0.59d0
       Nc=0.95d0
    ELSE
       F=1.d0
       c=0.69d0
       Nc=1.d0
    ENDIF
    Yb=IXYZb(2); Yw=IXYZw(2)
    La=Lw*Yb/Yw
    D=F*(1.d0-1.d0/3.6d0*exp(-(La+42.d0)/92.d0))
    k=1.d0/(5.d0*La+1.d0)
    FL=0.2d0*k**4*5.d0*La+0.1d0*(1.d0-k**4)**2*(5.d0*La)**third
    nbw=Yb/Yw
    Nbb=0.725d0/nbw**0.2d0!=0.725*(Yb/Yw)^-0.2
    z=1.48d0+sqrt(nbw)
    CALL operate_matrix3x3(mcat_fwd,IXYZw,LMSw)
    CALL operate_matrix3x3(mcat_fwd,IXYZwr,LMSwr)
    IF (cf_verbose>=2) THEN
       WRITE(*,'(A,9(1X,F11.6))') 'Ew,Lw,La=',Lw*pi,Lw,La
       WRITE(*,'(A,9(1X,F16.8))') 'F,c,Nc  =',F,c,Nc
       WRITE(*,'(A,9(1X,F16.8))') 'La,D,FL =',La,D,FL
!       WRITE(*,'(A,9(1X,F16.8))') 
       WRITE(*,'(A)') '************************************************************'
       WRITE(*,'(A)') 'GETRGB: Show Matrices'
       WRITE(*,'(A)') '        mcat_fwd'
       WRITE(*,'(A,3(1X,F16.12))') 'f11,f12,f13 =',(mcat_fwd(j,1),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'f21,f22,f23 =',(mcat_fwd(j,2),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'f31,f32,f33 =',(mcat_fwd(j,3),j=1,3)
       WRITE(*,*)
       WRITE(*,'(A)') '        mcat_bwd'
       WRITE(*,'(A,3(1X,F16.12))') 'b11,b12,b13 =',(mcat_bwd(j,1),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'b21,b22,b23 =',(mcat_bwd(j,2),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'b31,b32,b33 =',(mcat_bwd(j,3),j=1,3)
       WRITE(*,*)
       WRITE(*,'(A)') '        HPE_bwd'
       WRITE(*,'(A,3(1X,F16.12))') 'h11,h12,h13 =',(hpespace_bwd(j,1),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'h21,h22,h23 =',(hpespace_bwd(j,2),j=1,3)
       WRITE(*,'(A,3(1X,F16.12))') 'h31,h32,h33 =',(hpespace_bwd(j,3),j=1,3)
       WRITE(*,*)
    ENDIF
    RETURN
  END SUBROUTINE ciecam02_precalcs

  SUBROUTINE ciecam02_matops(mode,D,FL,w2wr,LMSw,LMSwr,IXYZ,IXYZc,LMS,LMSc,LMS1,LMSa)
!    USE color_functions_mathconst
!    USE chromadapt_matrices
    IMPLICIT NONE
    INTEGER :: mode
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMSw,LMSwr,IXYZ,IXYZc
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMS,LMSc,LMS1,LMSa
    REAL(KIND=CF_DOUBLE) :: D,fll,flm,fls,FL,w2wr
    IF (mode<=0) THEN!get LMS from IXYZ only if needed
       CALL operate_matrix3x3(mcat_fwd,IXYZ,LMS)
    ENDIF
    LMSc(1) =(w2wr*LMSwr(1)/LMSw(1)*D+1.d0-D)*LMS(1)
    LMSc(2) =(w2wr*LMSwr(2)/LMSw(2)*D+1.d0-D)*LMS(2)
    LMSc(3) =(w2wr*LMSwr(3)/LMSw(3)*D+1.d0-D)*LMS(3)
    CALL operate_matrix3x3(mcat_bwd,LMSc,IXYZc)
    CALL operate_matrix3x3(hpespace_fwd,IXYZc,LMS1)
    fll=sign(abs(FL*LMS1(1)/100.d0)**0.42d0,LMS1(1))
    flm=sign(abs(FL*LMS1(2)/100.d0)**0.42d0,LMS1(2))
    fls=sign(abs(FL*LMS1(3)/100.d0)**0.42d0,LMS1(3))
    LMSa(1)=400.d0*fll/(27.13d0+fll)+0.1d0
    LMSa(2)=400.d0*flm/(27.13d0+flm)+0.1d0
    LMSa(3)=400.d0*fls/(27.13d0+fls)+0.1d0
    IF (cf_verbose>=1) THEN
       WRITE(*,'(/A)') 'Matrix operations'
       WRITE(*,'(A,9(1X,F16.8))') 'fll,flm,fls=',fll,flm,fls
    ENDIF
  END SUBROUTINE ciecam02_matops

  SUBROUTINE ciecam02_fwd(IXYZi,IXYZbi,IXYZwi,IXYZwri,surround_condition,Lw,JCh,QMH,inpmode)
!  inpmode: 10-digit: 0: assume XYZ* as scaled to 100 (CIECAM02 default)
!                     1: assume XYZ* as scaled to unity (CIE1931 default)
!            1-digit: 0: assume angles in radians (math default)
!                     1: assume angles in degrees (CIECAM02 default)
!    USE color_functions_extended
!    USE chromadapt_matrices
    IMPLICIT NONE
    INTEGER :: i,j,ih360,ihq,ihp1,inpmode,degmode,flgxyzunity
    INTEGER :: surround_condition!0:dark,1:dim,2:average
!    EXTERNAL cf_safepower
!    REAL(KIND=CF_DOUBLE) :: cf_safepower!safe power function
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZi,IXYZbi,IXYZwi,IXYZwri
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ,IXYZb,IXYZw,IXYZwr,dum3
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMS,LMSb,LMSw,LMSwr!=(/1.d0,1.d0,1.d0/)
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMSc,LMScw,IXYZc,IXYZcw,LMS1,LMS1w,LMSa,LMSaw,JCh,QMH
    REAL(KIND=CF_DOUBLE) :: SR,Yb,Yw,Ywr,w2wr,Lw,La,F,c,Nc,Nbb,Ncb,D
    REAL(KIND=CF_DOUBLE) :: k,FL,fll,flm,fls,fllw,flmw,flsw,c1,c2,c3,a,b
    REAL(KIND=CF_DOUBLE) :: Jlight,hue_angle,hdeg,h1,dh1,Chroma,Hquad,Qbright,Mcolor,sat
    REAL(KIND=CF_DOUBLE) :: nbw,aprobe,awhite,z,et,ei,hi,hqi,hip1,eip1,tmp1,tmp2,tmp3
    REAL(KIND=CF_DOUBLE),DIMENSION(2,0:4) :: hdata
    flgxyzunity=inpmode/10
    degmode=inpmode-10*flgxyzunity
    IF (flgxyzunity<=0) THEN
       IXYZ=IXYZi; IXYZb=IXYZbi; IXYZw=IXYZwi; IXYZwr=IXYZwri
    ELSE
       IXYZ=1.d2*IXYZi; IXYZb=1.d2*IXYZbi; IXYZw=1.d2*IXYZwi; IXYZwr=1.d2*IXYZwri
    ENDIF
    CALL ciecam02_precalcs(surround_condition,Lw,IXYZb,IXYZw,IXYZwr,LMSw,LMSwr,Yb,Yw,nbw,Nbb,z,F,FL,LA,D,c,Nc,k)
    Ywr=IXYZwr(2)
    w2wr=Yw/Ywr
! Convert whites and test color into LMS
    CALL operate_matrix3x3(mcat_fwd,IXYZ,LMS)
    CALL ciecam02_matops(0,D,FL,w2wr,LMSw,LMSwr, IXYZ, IXYZc, LMS, LMSc, LMS1, LMSa)
    CALL ciecam02_matops(1,D,FL,w2wr,LMSw,LMSwr, IXYZw,IXYZcw,LMSw,LMScw,LMS1w,LMSaw)
    c1=LMSa(1)-LMSa(2)
    c2=LMSa(2)-LMSa(3)
    c3=LMSa(3)-LMSa(1)
    a=c1-c2/11.d0
    b=(c2-c3)/9.d0!0.5*(c2-c1+c1-c3)/4.5
    hue_angle=atan2(b,a)
    IF (hue_angle<0.d0) hue_angle=hue_angle+tau
    hdeg=hue_angle*r2d
    dh1=hdeg-hq_table(1,1)
    ih360=floor(dh1/360.d0)
    h1=hdeg-360.d0*ih360
    DO i=0,4
       hdata(1,i)=hq_table(1,i+1)
    ENDDO
    CALL cf_tablookup(0,4,hdata,h1,tmp1,tmp2,i)!tmp's just dummies here
    ihq=mod(i,5)+1
    ihp1=ihq+1
    IF (ihp1>5) ihp1=ihp1-5
!    WRITE(*,*) 'dh1,ih36,ihq=',dh1,ih360,ihq
    hi =hq_table(1,ihq); hip1=hq_table(1,ihp1)
    ei =hq_table(2,ihq); eip1=hq_table(2,ihp1)
    hqi=hq_table(3,ihq)
    tmp1=(h1-hi)/ei
    tmp2=(hip1-h1)/eip1
    Hquad=hqi+100.d0*tmp1 / (tmp1+tmp2)
    Hquad=Hquad-400.d0*floor(Hquad/400.d0)
    et=0.25d0*(cos(hue_angle+2.d0)+3.8d0)!note: h is in radians here
!    nbw=Yb/Yw
!    Nbb=0.725d0/nbw**0.2d0!=0.725*(Yb/Yw)^-0.2
    Ncb=Nbb
    aprobe=(2.d0*LMSa(1)+LMSa(2)+0.05d0*LMSa(3)-0.305d0)*Nbb
    awhite=(2.d0*LMSaw(1)+LMSaw(2)+0.05d0*LMSaw(3)-0.305d0)*Nbb
    z=1.48d0+sqrt(nbw)
!    Jlight=100.d0*(aprobe/awhite)**(c*z)
    Jlight=100.d0*cf_safepower(aprobe/awhite,c*z)
    tmp1=50.d3/13.d0*Nc*Ncb*et*sqrt(a*a+b*b)/(LMSa(1)+LMSa(2)+1.05d0*LMSa(3))
    tmp2=sqrt(max(Jlight/100.d0,0.d0))!truncated at zero
    tmp3=FL**0.25d0
!    Chroma=tmp1**0.9d0*tmp2*(1.64d0-0.29d0**nbw)**0.73d0
    Chroma=cf_safepower(tmp1,0.9d0)*tmp2*cf_safepower(1.64d0-0.29d0**nbw,0.73d0)
    Qbright=4.d0/c*tmp2*(awhite+4.d0)*tmp3
    Mcolor=Chroma*tmp3
    sat=100.d0*sqrt(Mcolor/Qbright)
    IF (degmode>=1) THEN!input in degrees
       JCh=(/Jlight,Chroma,hue_angle*r2d/)
    ELSE!input in radians
       JCh=(/Jlight,Chroma,hue_angle/)
    ENDIF
    QMH=(/Qbright,Mcolor,Hquad/)
    IF (cf_verbose>=1) THEN
       WRITE(*,'(/A)') 'Forward'
       WRITE(*,'(A,9(4X,I3))') 'in,d,100=',inpmode,degmode,flgxyzunity
       WRITE(*,'(A,9(1X,F16.8))') 'Lw,c*z  =',Lw,c*z
       WRITE(*,'(A,9(1X,F16.8))') 'Jlight  =',Jlight
       WRITE(*,'(A,9(1X,F16.8))') 'tmp123  =',tmp1,tmp2,tmp3
       WRITE(*,'(A,9(1X,F16.8))') 'Yb,Yw,Yr=',Yb,Yw,Ywr
       WRITE(*,'(A,9(1X,F16.8))') 'F,c,Nc  =',F,c,Nc
       WRITE(*,'(A,9(1X,F16.8))') 'La,D,FL =',La,D,FL
       WRITE(*,'(A,9(1X,F16.8))') 'Awhite  =',Awhite
       WRITE(*,'(A,9(1X,F16.8))') 'A,C,h   =',Aprobe,Chroma,hue_angle*r2d
       WRITE(*,'(A,9(1X,F16.8))') 'a,b,Ncb =',a,b,Ncb
       WRITE(*,'(A,9(1X,F16.8))') 'k,z,n   =',k,z,nbw
       WRITE(*,'(A,9(1X,F16.8))') 'Nc,Nbb  =',Nc,Nbb
       WRITE(*,'(A,9(1X,F16.8))') 'w2wr    =',w2wr
       WRITE(*,'(A,9(1X,F16.8))') 'et,tmp1 =',et,tmp1
       WRITE(*,'(A,9(1X,F16.8))') 'IXYZc   =',IXYZc
       WRITE(*,'(A,9(1X,F16.8))') 'LMS     =',LMS
       WRITE(*,'(A,9(1X,F16.8))') 'LMSw    =',LMSw
       WRITE(*,'(A,9(1X,F16.8))') 'LMSc    =',LMSc
       WRITE(*,'(A,9(1X,F16.8))') 'LMS1    =',LMS1
       WRITE(*,'(A,9(1X,F16.8))') 'LMS1w   =',LMS1w
       WRITE(*,'(A,9(1X,F16.8))') 'LMSa    =',LMSa
       WRITE(*,'(A,9(1X,F16.8))') 'LMSaw   =',LMSaw
       WRITE(*,'(A,9(1X,F16.8))') 'JCh     =',JCh
       WRITE(*,'(A,9(1X,F16.8))') 'QMH     =',QMH
       WRITE(*,'(A,1X,F16.8,I4)') 'dh1,ih36=',dh1,ih360
       WRITE(*,'(A,9(1X,F16.8))') 'hdeg,h1 =',hdeg,h1
       WRITE(*,'(A,9(1X,F16.8))') 'hi,ei,Hi=',hi,ei,Hqi
       WRITE(*,*)
!       IF (cf_verbose>=2) THEN
!          WRITE(*,'(A)') '************************************************************'
!          WRITE(*,'(A)') 'GETRGB: Show Matrices'
!          WRITE(*,'(A)') '        mcat_fwd'
!          WRITE(*,'(A,3(1X,F16.12))') 'a11,a12,a13 =',(mcat_fwd(j,1),j=1,3)
!          WRITE(*,'(A,3(1X,F16.12))') 'a21,a22,a23 =',(mcat_fwd(j,2),j=1,3)
!          WRITE(*,'(A,3(1X,F16.12))') 'a31,a32,a33 =',(mcat_fwd(j,3),j=1,3)
!          WRITE(*,*)
!          WRITE(*,'(A)') '        mcat_bwd'
!          WRITE(*,'(A,3(1X,F16.12))') 'b11,b12,b13 =',(mcat_bwd(j,1),j=1,3)
!          WRITE(*,'(A,3(1X,F16.12))') 'b21,b22,b23 =',(mcat_bwd(j,2),j=1,3)
!          WRITE(*,'(A,3(1X,F16.12))') 'b31,b32,b33 =',(mcat_bwd(j,3),j=1,3)
!          WRITE(*,*)
!       ENDIF
    ENDIF
    RETURN
  END SUBROUTINE ciecam02_fwd

  SUBROUTINE ciecam02_bwd(JChQMH,qmode,IXYZo,IXYZbi,IXYZwi,IXYZwri,surround_condition,Lw,inpmode)
!  JChQMH : Array consisting eiher of JCh, JCH or QMH for qmode=0, 1 or 2, respectively.
!  inpmode: 10-digit: 0: assume XYZ* as scaled to 100 (CIECAM02 default)
!                     1: assume XYZ* as scaled to unity (CIE1931 default)
!            1-digit: 0: assume angles in radians (math default)
!                     1: assume angles in degrees (CIECAM02 default)
!    USE color_functions_extended
!    USE chromadapt_matrices
    IMPLICIT NONE
    INTEGER :: qmode,i,j,ihq,ihp1,ih360,inpmode,degmode,flgxyzunity
    INTEGER :: surround_condition!0:dark,1:dim,2:average
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: JChQMH,IXYZo,IXYZbi,IXYZwi,IXYZwri
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ,IXYZb,IXYZw,IXYZwr
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMS,LMSb,LMSw,LMSwr!=(/1.d0,1.d0,1.d0/)
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: LMSc,LMScw,IXYZc,IXYZcw,LMS1,LMS1w,LMSa,LMSaw,JCh,QMH
    REAL(KIND=CF_DOUBLE) :: SR,Yb,Yw,Ywr,w2wr,Lw,La,F,c,Nc,Nbb,Ncb,D
    REAL(KIND=CF_DOUBLE) :: k,FL,fll,flm,fls,fllw,flmw,flsw,c1,c2,c3,a,b,FL4th,tmp1,tmp2,hi,ei,Hqi,hip1,eip1
    REAL(KIND=CF_DOUBLE) :: Jlight,hue_angle,Chroma,Hquad,Qbright,Mcolor,sat,hdeg
    REAL(KIND=CF_DOUBLE) :: nbw,aprobe,awhite,z,et,t,p1,p2,p3,p4,p5,chue,shue,ram1,gam1,bam1
    REAL(KIND=CF_DOUBLE) :: w460=460.d0/1403.d0,w220=220.d0/1403.d0,w27=27.d0/1403.d0,w6300=6300.d0/1403.d0
    REAL(KIND=CF_DOUBLE) :: w451=451.d0/1403.d0,w288=288.d0/1403.d0,w891=891.d0/1403.d0,w261=261.d0/1403.d0
    REAL(KIND=CF_DOUBLE),DIMENSION(2,0:4) :: hdata
    flgxyzunity=inpmode/10
    degmode=inpmode-10*flgxyzunity
    IF (flgxyzunity<=0) THEN
       IXYZb=IXYZbi; IXYZw=IXYZwi; IXYZwr=IXYZwri
    ELSE
       IXYZb=1.d2*IXYZbi; IXYZw=1.d2*IXYZwi; IXYZwr=1.d2*IXYZwri
    ENDIF
    CALL ciecam02_precalcs(surround_condition,Lw,IXYZb,IXYZw,IXYZwr,LMSw,LMSwr,Yb,Yw,nbw,Nbb,z,F,FL,LA,D,c,Nc,k)
    Ywr=IXYZwr(2)
    w2wr=Yw/Ywr
!    nbw=Yb/Yw
    CALL ciecam02_matops(1,D,FL,w2wr,LMSw,LMSwr, IXYZw,IXYZcw,LMSw,LMScw,LMS1w,LMSaw)
    awhite=(2.d0*LMSaw(1)+LMSaw(2)+0.05d0*LMSaw(3)-0.305d0)*Nbb
!
    IF (qmode>=1) THEN
       Hquad=JChQMH(3)
       Hquad=Hquad-400.d0*floor(Hquad/400.d0)
       IF (qmode==2) THEN!Input = QMH
          Qbright=JChQMH(1); Mcolor=JChQMH(2)
          Mcolor=max(Mcolor,0.d0)
          FL4th=FL**0.25d0
          Jlight=6.25d0*(c*Qbright/((awhite+4.d0)*FL4th))**2
          Chroma=Mcolor/FL4th
       ELSE
          Jlight=JChQMH(1); Chroma=JChQMH(2)
!          WRITE(*,*) 'Chroma=',Chroma
       ENDIF
       DO i=0,4
          hdata(1,i)=hq_table(3,i+1)
       ENDDO
       CALL cf_tablookup(0,4,hdata,Hquad,tmp1,tmp2,i)!tmp's just dummies here
       ihq=mod(i,5)+1
       ihp1=ihq+1
       IF (ihp1>5) ihp1=ihp1-5
!       WRITE(*,*) 'Hquad,i,ihq=',Hquad,i,ihq
       hi =hq_table(1,ihq); hip1=hq_table(1,ihp1)
       ei =hq_table(2,ihq); eip1=hq_table(2,ihp1)
       Hqi=hq_table(3,ihq)
       hdeg= ((Hquad-Hqi)*(eip1*hi-ei*hip1)-100.d0*hi*eip1) &
            /((Hquad-Hqi)*(eip1-ei)-100.d0*eip1)
       JCh=(/Jlight,Chroma,hdeg-360.d0*floor(hdeg/360.d0)/)!in degrees as default
    ELSE
       Jlight=JChQMH(1); Chroma=JChQMH(2)
       Chroma=max(Chroma,0.d0)
       IF (degmode>=1) THEN!input in degrees
          hdeg=JChQMH(3)
       ELSE!input in radians
          hdeg=JChQMH(3)*r2d
       ENDIF
    ENDIF
    ih360=floor(hdeg/360.d0)
    hdeg =hdeg-ih360*360.d0
    hue_angle=hdeg/r2d
    t=(Chroma/(sqrt(Jlight/1.d2)*(1.64d0-0.29d0**nbw)**0.73d0))**(1.d0/0.9d0)
    et=(cos(hue_angle+2.d0)+3.8d0)/4.d0
    aprobe=awhite*(Jlight/1.d2)**(1.d0/(c*z))
    IF (t<=0.d0) THEN
       p1=1.d99
    ELSE
       p1=(50000.d0/13.d0)*Nc*Nbb*et/t
    ENDIF
    p2=Aprobe/Nbb+0.305d0
    p3=1.05d0
    chue=cos(hue_angle); shue=sin(hue_angle)
!    w460=460.d0/1403.d0; w220=220.d0/1403.d0; w27=27.d0/1403.d0; w6300=6300.d0/1403.d0
    IF (abs(shue)>=abs(chue)) THEN
       p4=p1/shue
       b=p2*(2.d0+p3)*w460/(p4+(2.d0+p3)*w220*chue/shue-w27+p3*w6300)
       a=b*chue/shue
       IF (cf_verbose>=1) WRITE(*,'(/A)') '|sin hrad| >= |cos hrad|'
    ELSE
       p5=p1/chue
! Errors in Yang Xue's thesis, Eq. 2-27, part a(p5):
! 1. It should be sin h / cos h, not the other way round.
! 2. The sign before p3*6300/1403 must be negative.
       a=p2*(2.d0+p3)*w460/(p5+(2.d0+p3)*w220-(w27-p3*w6300)*shue/chue)
       b=a*shue/chue
       IF (cf_verbose>=1) WRITE(*,'(/A)') '|sin hrad| < |cos hrad|'
    ENDIF
    LMSa(1)=w460*p2+w451*a+w288*b
    LMSa(2)=w460*p2-w891*a-w261*b
    LMSa(3)=w460*p2-w220*a-w6300*b
    ram1=LMSa(1)-0.1d0
    gam1=LMSa(2)-0.1d0
    bam1=LMSa(3)-0.1d0
    LMS1(1)=1.d2/FL*((27.13d0*abs(ram1))/(4.d2-abs(ram1)))**(1.d0/0.42d0) * sign(1.d0,ram1)
    LMS1(2)=1.d2/FL*((27.13d0*abs(gam1))/(4.d2-abs(gam1)))**(1.d0/0.42d0) * sign(1.d0,gam1)
    LMS1(3)=1.d2/FL*((27.13d0*abs(bam1))/(4.d2-abs(bam1)))**(1.d0/0.42d0) * sign(1.d0,bam1)
    CALL operate_matrix3x3(hpespace_bwd,LMS1,IXYZc)
    CALL operate_matrix3x3(mcat_fwd,IXYZc,LMSc)
    LMS(1)=LMSc(1)/(w2wr*LMSwr(1)/LMSw(1)*D+1.d0-D)
    LMS(2)=LMSc(2)/(w2wr*LMSwr(2)/LMSw(2)*D+1.d0-D)
    LMS(3)=LMSc(3)/(w2wr*LMSwr(3)/LMSw(3)*D+1.d0-D)
    CALL operate_matrix3x3(mcat_bwd,LMS,IXYZ)
    IF (flgxyzunity<=0) THEN
       IXYZo=IXYZ!nothing to change
    ELSE
       IXYZo=IXYZ/1.d2!rescale to unity
    ENDIF
    IF (cf_verbose>=1) THEN
       WRITE(*,'(/A)') 'Backward'
       WRITE(*,'(A,9(4X,I3))') 'in,d,100=',inpmode,degmode,flgxyzunity
       WRITE(*,'(A,9(1X,F24.16))') 'IXYZw  =',IXYZw
       WRITE(*,'(A,9(1X,F24.16))') 'IXYZwr =',IXYZw
       WRITE(*,'(A,9(1X,F24.16))') 'IXYZb  =',IXYZb
       WRITE(*,'(A,9(1X,F16.8))') 'Lw      =',Lw
       WRITE(*,'(A,9(1X,E16.8))') 'JChQMH  =',JChQMH
       WRITE(*,'(A,9(1X,F16.8))') 'JCh     =',JCh
       WRITE(*,'(A,9(1X,F16.8))') 'ch,sh   =',chue,shue
!       WRITE(*,'(A,9(1X,F16.8))') 'w460,w451,w288 =',w460,w451,w288
!       WRITE(*,'(A,9(1X,F16.8))') 'w460,w891,w261 =',w460,w891,w261
!       WRITE(*,'(A,9(1X,F16.8))') 'w460,w220,w6300=',w460,w220,w6300
       WRITE(*,'(A,9(1X,F16.8))') 'Yb,Yw,Yr=',Yb,Yw,Ywr
       WRITE(*,'(A,9(1X,F16.8))') 'F,c,Nc  =',F,c,Nc
       WRITE(*,'(A,9(1X,E16.8))') 'La,D,FL =',La,D,FL
       WRITE(*,'(A,9(1X,F16.8))') 'A,C,h   =',Aprobe,Chroma,hue_angle*r2d
       WRITE(*,'(A,9(1X,F16.8))') 'p1,p2,p3=',p1,p2,p3
       WRITE(*,'(A,9(1X,F16.8))') 'a,b,Ncb =',a,b,Ncb
       WRITE(*,'(A,9(1X,F16.8))') 'k,z,n   =',k,z,nbw
       WRITE(*,'(A,9(1X,F16.8))') 'Nc,Nbb  =',Nc,Nbb
       WRITE(*,'(A,9(1X,F16.8))') 'w2wr    =',w2wr
       WRITE(*,'(A,9(1X,F16.8))') 'et,t    =',et,t
       WRITE(*,'(A,9(1X,F16.8))') '4.d2-abs(ram1)=',(4.d2-abs(ram1))
       WRITE(*,'(A,9(1X,F16.8))') '4.d2-abs(gam1)=',(4.d2-abs(gam1))
       WRITE(*,'(A,9(1X,F16.8))') '4.d2-abs(bam1)=',(4.d2-abs(bam1))
!       WRITE(*,'(A,9(1X,E16.8))') 'term 1 =',((27.13d0*abs(ram1))/(4.d2-abs(ram1)))**(1.d0/0.42d0)
!       WRITE(*,'(A,9(1X,E16.8))') 'term 1 =',((27.13d0*abs(gam1))/(4.d2-abs(gam1)))**(1.d0/0.42d0)
!       WRITE(*,'(A,9(1X,E16.8))') 'term 1 =',((27.13d0*abs(bam1))/(4.d2-abs(bam1)))**(1.d0/0.42d0)
       WRITE(*,'(A,9(1X,F16.8))') 'IXYZc   =',IXYZc
       WRITE(*,'(A,9(1X,F16.8))') 'LMSa    =',LMSa
!       WRITE(*,'(A,9(1X,F16.8))') 'LMSaw   =',LMSaw
       WRITE(*,'(A,9(1X,F16.8))') 'LMS1    =',LMS1
!       WRITE(*,'(A,9(1X,F16.8))') 'LMS1w   =',LMS1w
       WRITE(*,'(A,9(1X,F16.8))') 'LMSc    =',LMSc
       WRITE(*,'(A,9(1X,F16.8))') 'LMS     =',LMS
!       WRITE(*,'(A,9(1X,F16.8))') 'LMSw    =',LMSw
       WRITE(*,'(A,1X,F16.8,I4)') 'hd,ih360=',hdeg,ih360
       WRITE(*,'(A,9(1X,F24.16))') 'IXYZo   =',IXYZo
       WRITE(*,*)
!       stop
    ENDIF
    RETURN
  END SUBROUTINE ciecam02_bwd

  SUBROUTINE euc2ciecam(JCh,JChnew,direction)
!    USE color_functions
    IMPLICIT NONE
    INTEGER :: direction
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: JCh,JChnew
    REAL(KIND=CF_DOUBLE) :: J,C,h,Jnew,Cnew,hnew, kj=1.d0,kc=1.d0,kh=1.d0
    REAL(KIND=CF_DOUBLE) :: aux1=126.4911064067351733d0,aux2=1.581138830084189666d-2
    IF (direction>=1) THEN
! JCh-JChnew
       J=JCh(1); C=JCh(2); h=JCh(3)
       Jnew=kj*0.99d0*aux1*atan(aux2*J)
       Cnew=kc*47.d0*log(1.d0+0.02d0*C)
       hnew=h
       JChnew=(/Jnew,Cnew,hnew/)
    ELSE IF (direction<=-1) THEN
! JChnew-JCh
       Jnew=JChnew(1); Cnew=JChnew(2); hnew=JChnew(3)
       J=tan(Jnew/(kj*0.99d0*aux1)) / aux2
       C=50.d0*(exp(Cnew/(kc*47.d0))-1.d0)
       h=hnew
       JCh=(/J,C,h/)
    ENDIF
    IF (cf_verbose>=1) THEN
       IF (direction>=1) THEN
          WRITE(*,'(/A)') 'Forward'
          WRITE(*,'(A,9(1X,F16.8))') 'JCh   =',JCh
          WRITE(*,'(A,9(1X,F16.8))') 'JChnew=',JChnew
       ELSE IF (direction<=-1) THEN
          WRITE(*,'(/A)') 'Backward'
          WRITE(*,'(A,9(1X,F16.8))') 'JChnew=',JChnew
          WRITE(*,'(A,9(1X,F16.8))') 'JCh   =',JCh
       ELSE
          WRITE(*,'(/A)') 'Nothing'
       ENDIF
       WRITE(*,'(A,9(1X,F16.8))') '1+0.02*C=',1.d0+0.02d0*C
    ENDIF
    RETURN
  END SUBROUTINE euc2ciecam

  SUBROUTINE ciecam02_xyzdiff(IXYZi1,IXYZi2,IXYZbi,IXYZwi,IXYZwri,surround_condition,Lw,DE02,c02mode,inpmode)
    IMPLICIT NONE
    INTEGER :: c02mode,surround_condition,inpmode,degmode,idum
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZi1,IXYZi2,IXYZbi,IXYZwi,IXYZwri
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: JCh1,QMH1,JCh2,QMH2
    REAL(KIND=CF_DOUBLE) :: Lw,DE02
    idum=inpmode/10
    degmode=inpmode-10*idum
    CALL ciecam02_fwd(IXYZi1,IXYZbi,IXYZwi,IXYZwri,surround_condition,Lw,JCh1,QMH1,inpmode)
    CALL ciecam02_fwd(IXYZi2,IXYZbi,IXYZwi,IXYZwri,surround_condition,Lw,JCh2,QMH2,inpmode)
    CALL ciecam02_diff(JCh1,QMH1,JCh2,QMH2,DE02,c02mode,degmode)
    RETURN
  END SUBROUTINE ciecam02_xyzdiff

  SUBROUTINE ciecam02_diff(JCh1,QMH1,JCh2,QMH2,DE02,c02mode,degmode)
    IMPLICIT NONE
    INTEGER :: degmode,c02mode!0:UCS, 1:LCD, 2:SCD
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: JCh1,QMH1,JCh2,QMH2,Jab1,Jab2,coeffs
    REAL(KIND=CF_DOUBLE) :: DE02,kl,c1=7.d-3,c2
    REAL(KIND=CF_DOUBLE) :: J,M,h,JP1,MP1,JP2,MP2,ap1,bp1,ap2,bp2,kdeg,krad
    CALL ciecam02_JMh2Jab(JCh1,QMH1,Jab1,coeffs,c02mode,degmode); JP1=Jab1(1); ap1=Jab1(2); bp1=Jab1(3)
    CALL ciecam02_JMh2Jab(JCh2,QMH2,Jab2,coeffs,c02mode,degmode); JP2=Jab2(1); ap2=Jab2(2); bp2=Jab2(3)
    kl=coeffs(1); c1=coeffs(2); c2=coeffs(3)
    DE02=sqrt(((JP2-JP1)/kl)**2+(ap2-ap1)**2+(bp2-bp1)**2)
    RETURN
  END SUBROUTINE ciecam02_diff

  SUBROUTINE ciecam02_JMh2Jab(JChi,QMHi,Jab,coeffs,c02mode,degmode)
    IMPLICIT NONE
!   INPUT:
!   JCh,QMH : Cylinder coordinates JCh and QMH
!
!   OUTPUT:
!   Jab     : Cartesian coordinates
!   coeffs  : kl,c1,c2
!
!   Settings:
!   c02mode = 0: CIECAM02 UCS
!          1: CIECAM02 LCD
!          2: CIECAM02 SCD
!         -1: CIECAM02 EUC
!         -2: CIECAM02 Hquad
!         -3: CIECAM02 Hquad EUC
    INTEGER :: degmode,c02mode!0:UCS, 1:LCD, 2:SCD
    REAL(KIND=CF_DOUBLE),DIMENSION(3) :: JCh,QMH,JChi,QMHi,Jab,JChnew,coeffs
    REAL(KIND=CF_DOUBLE) :: DE02,kl,c1=7.d-3,c2
    REAL(KIND=CF_DOUBLE) :: J,M,h,JP,MP,ap,bp,Cp,kdeg,krad
    IF (degmode>=1) THEN!input in degrees
       kdeg=1.d0
       krad=d2r
    ELSE
       kdeg=r2d
       krad=1.d0
    ENDIF
    IF (c02mode==1) THEN
       kl=0.77d0; c2=0.0053d0
    ELSE IF (c02mode==2) THEN
       kl=1.24d0; c2=0.0363d0
    ELSE
       kl=1.d0; c2=0.0228d0
    ENDIF
    coeffs=(/kl,c1,c2/)
    JCh=JCHi; QMH=QMHi
    IF (c02mode==-1) THEN! Euclidian mode
       CALL euc2ciecam(JCh,JChnew,1)
       JP=JChnew(1);CP=JChnew(2); h=JChnew(3)*krad
       ap=Cp*cos(h)
       bp=Cp*sin(h)
    ELSE IF (c02mode==-2) THEN! JMHquad mode
       JP=JCh(1);CP=JCh(2); h=QMH(3)*krad*0.9d0
       ap=Cp*cos(h)
       bp=Cp*sin(h)
    ELSE IF (c02mode==-3) THEN! Euclidian JMHquad mode
       CALL euc2ciecam(JCh,JChnew,1)
       JP=JChnew(1);CP=JChnew(2); h=QMH(3)*krad*0.9d0
       ap=Cp*cos(h)
       bp=Cp*sin(h)
    ELSE!CIE c02mode 0--2
       J=JCh(1);M=QMH(2); h=JCh(3)*krad
       JP=(1.d0+100.d0*c1)*J/(1.d0+c1*J)
       MP=log(1.d0+c2*M)/c2
       ap=MP*cos(h)
       bp=MP*sin(h)
    ENDIF
    Jab=(/Jp,ap,bp/)
    RETURN
  END SUBROUTINE ciecam02_JMh2Jab

  SUBROUTINE cf_tablookup(mode,ndata,adata,x,y,dydx,k)
!---- Simple linear interpolation of tabulated data
!**** INPUT
!           mode : <=0 - determine table index only
!                   >0 - also interpolate data
!          ndata : Number of probes
! adata(1:2,0:n) : Table of x and y calues
!                  (1,i) = xa(i) abscissae
!                  (2,i) = ya(i) ordinates
!              x : Actual abscissa
!**** OUTPUT
!              y : Interpolated value of y
!           dydx : First derivative (slope) of y
    IMPLICIT NONE
    INTEGER :: i,k,k0,k1,mode,ndata,order
    DOUBLE PRECISION :: adata(2,0:ndata),x,y,dydx
    DOUBLE PRECISION :: c,d,s,xmin,xmax
    k0 = 0
    k1 = ndata
    IF (adata(1,ndata).lt.adata(1,0)) THEN
       order=-1
    ELSE
       order=1
    ENDIF
!      write(*,*) 'ndata,x =',ndata,x
!====== Bracketing
!---- Zero outside
    xmin=min(adata(1,0),adata(1,ndata))
    xmax=max(adata(1,0),adata(1,ndata))
    IF (x.lt.xmin.OR.x.gt.xmax) THEN
       y=0.d0
       dydx=0.d0
       RETURN
    ENDIF
!---- Search table
1   IF (k1-k0 .gt. 1) THEN
       k = (k1+k0)/2
!         write(*,*) 'k0,k1 =',k0,k1,adata(1,k0),adata(1,k1),
!     &        adata(2,k0),adata(2,k1)
       IF (order*adata(1,k).gt.order*x) THEN
          k1 = k
       ELSE
          k0 = k
       ENDIF
       GOTO 1
    ENDIF
    k=k0
    IF (mode<=0) RETURN!skip interpolation
!      kexpt=k
!      write(*,*) 'k0,k1 =',k0,k1,adata(1,k0),adata(1,k1)
!====== Linear Interpolation
    c=(x-adata(1,k0))
    d=(adata(1,k1)-adata(1,k0))!difference of x
    s=(adata(2,k1)-adata(2,k0))/d!slope = dy/dx = dlgt/dx
    y=c*s+adata(2,k0)
!      write(*,*) 'x,y,k0,k1 =',x,y,k0,k1
    RETURN
  END SUBROUTINE cf_tablookup

END MODULE color_functions_ciecam02

!===============================================================================
!==== Extended functions (DIN99d+o) ============================================
!===============================================================================

MODULE color_functions_extended
  USE color_functions_mathconst
  USE color_functions
  REAL(KIND=CF_DOUBLE) :: kE_DIN99=1.d0,kCH_DIN99=1.d0
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: euc_mat_rot=RESHAPE((/1.d0, 0.d0,     0.d0, &
                                                               0.d0, 1.d0,    -0.1474d0, &
                                                               0.d0,-0.0265d0, 0.5507d0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: euc_mat_red=RESHAPE((/1.d0, 0.d0,     0.d0, &
                                                               0.d0, 0.9721d0, 0.2347d0, &
                                                               0.d0,-0.2347d0, 0.9721d0/),(/3,3/))
  REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: euc_mat_scal=RESHAPE((/100.d0,   0.d0,   0.d0, &
                                                                  0.d0, 150.d0,   0.d0, &
                                                                  0.d0,   0.d0, 150.d0/),(/3,3/))

  CONTAINS

    SUBROUTINE lab2lch(Lab,LCh,direction,degmode)
      IMPLICIT NONE
      INTEGER direction,degmode
      REAL(KIND=CF_DOUBLE) :: Lab(3),LCh(3),L,a,b,C,h,xdum
      IF (direction>=1) THEN!L*a*b* to L*C*h*
         L=Lab(1);a=Lab(2);b=Lab(3)
         C=sqrt(a**2+b**2)
         IF (C.gt.0.d0) THEN
            xdum=max(min(a/(C),1.d0),-1.d0)
         ELSE
            xdum=0.d0
         ENDIF
         h=acos(xdum)*sign(1.d0,b)
         IF (h<0.d0) THEN
            h=h+tau
         ELSE IF (h>tau) THEN
            h=h-tau
         ENDIF
         IF (degmode>=1) THEN
            h=h*r2d
         ENDIF
         LCh=(/L,C,h/)
      ELSE! L*C*h* to L*a*b*
         L=LCh(1);C=LCh(2);h=LCh(3)
         IF (degmode>=1) THEN
            h=h/r2d
         ENDIF
         a=C*cos(h)
         b=C*sin(h)
         Lab=(/L,a,b/)
      ENDIF
      RETURN
    END SUBROUTINE lab2lch

!-- CIEDE2000 Color difference formula

    SUBROUTINE CIEDE2000(Lab1,Lab2,DE00)
!      USE color_functions_extended
!      USE color_functions_mathconst
      IMPLICIT NONE
      DOUBLE PRECISION :: Lab1(3),Lab2(3),DE00, LCh1(3),LCh2(3),L1,L2,a1,a2,b1,b2,C1,C2,h1,h2, dl00
      DOUBLE PRECISION :: Lbar,Cbar,a100,b100,a200,b200, C100,C200, h100,h200, C00bar,dC00, dh00
      DOUBLE PRECISION :: SL,SC,SH,G00,T00,RT
      DOUBLE PRECISION :: hubar,dhu,pdum,ddum
      DOUBLE PRECISION :: kL=1.d0,kC=1.d0,kH=1.d0
!      CALL lab2lch(Lab1,LCh1,1,1)
!      CALL lab2lch(Lab2,LCh2,1,1)
      L1=Lab1(1); a1=Lab1(2); b1=Lab1(3); C1=sqrt(a1**2+b1**2)!; C1=LCh1(2); h1=LCh1(3)
      L2=Lab2(1); a2=Lab2(2); b2=Lab2(3); C2=sqrt(a2**2+b2**2)!; C2=LCh2(2); h2=LCh2(3)
      dl00=L2-L1
      Lbar=(L1+L2)/2.d0
      Cbar=(C1+C2)/2.d0
      G00=0.5d0*(1.d0-sqrt(Cbar**7/(Cbar**7+6.103515625d9)))!25^7=6103515625
      a100=a1*(1.d0+G00)
      a200=a2*(1.d0+G00)
      C100=sqrt(a100**2+b1**2)!a00 with CIELAB b
      C200=sqrt(a200**2+b2**2)!a00 with CIELAB b
      dC00=C200-C100
      C00bar=(C100+C200)/2.d0
      IF (b1==0.d0.AND.a100==0.d0) THEN
         h100=0.d0
      ELSE
         h100=atan2(b1,a100)!CIELAB b with a100; degrees used later
      ENDIF
      IF (b2==0.d0.AND.a200==0.d0) THEN
         h200=0.d0
      ELSE
         h200=atan2(b2,a200)
      ENDIF
      IF (h100<0.d0) h100=h100+tau
      IF (h200<0.d0) h200=h200+tau
      dh00=h200-h100
      IF (dh00>pi) THEN
         IF (h200<=h100) THEN
            dh00=dh00+tau
         ELSE
            dh00=dh00-tau
         ENDIF
      ENDIF
      dhu=2.d0*sqrt(C100*C200)*sin(dh00/2.d0)
      IF (C100*C200==0.d0) THEN
         hubar=h100+h200!no division be 2 here
!         WRITE(*,*) 'Case C1*C2=0'
      ELSE IF (abs(h100-h200)>pi.AND.h100+h200<tau) THEN!really h1'-h2'
         hubar=(h100+h200+tau)/2.d0
!         WRITE(*,*) 'Case h1-h2>pi, h1+h2<2pi'
      ELSE IF (abs(h100-h200)>pi.AND.h100+h200>=tau) THEN!really h1'-h2'
         hubar=(h100+h200-tau)/2.d0
!         WRITE(*,*) 'Case h1-h2>pi, h1+h2>=2pi'
      ELSE
         hubar=(h100+h200)/2.d0
!         WRITE(*,*) 'Case h1-h2<=pi'
      ENDIF
!   pi/6 = 30°
      T00=1.d0-0.17d0*cos(hubar-pi6)          +0.24d0*cos(2.d0*hubar) &
              +0.32d0*cos(3.d0*hubar+pi/30.d0)-0.20d0*cos(4.d0*hubar-pi*0.35d0)
      SL=1.d0+1.5d-2*(Lbar-5.d1)**2/sqrt(2.d1+(Lbar-5.d1)**2)
      SC=1.d0+4.5d-2*C00bar
      SH=1.d0+1.5d-2*C00bar*T00
      pdum=(hubar*r2d-275.d0)/25.d0
      RT=-2.d0*sqrt(C00bar**7/(C00bar**7+6.103515625d9))*sin(pi3*exp(-pdum**2))
      ddum=(dL00/kL/SL)**2 + (dC00/kC/SC)**2 + (dhu/kH/SH)**2 + RT*dC00/SC*dhu/SH
      DE00=sqrt(ddum)
      IF (cf_verbose>=1) THEN
         WRITE(*,'(A,9(1X,F12.6))') 'a,C,h1 =',a100,C100,r2d*h100,r2d*hubar,G00,T00,SL,SC,SH
         WRITE(*,'(A,9(1X,F12.6))') 'a,C,h2 =',a200,C200,r2d*h200,RT,DE00
!         WRITE(*,'(A,9(1X,F12.6))') 'h100'
      ENDIF
      RETURN
    END SUBROUTINE CIEDE2000

!-- DIN99d formulae, retrieved from thesis by Yang Zue (2008) and Shizhe Shen (2009)

    SUBROUTINE XYZ2DIN99d(Ixyz,Ixyzwhite,Lab99d,LCh99d,degmode)
      IMPLICIT NONE
      INTEGER :: degmode
      REAL(KIND=CF_DOUBLE) :: scal_L99d=325.2194339489675351d0
      REAL(KIND=CF_DOUBLE) :: Ix,Iy,Iz,Ls,as,bs,L99d,C99d,h99d,a99d,b99d
      REAL(KIND=CF_DOUBLE) :: e,f,G,hef
      REAL(KIND=CF_DOUBLE) :: d50=0.87266462599716479d0
      REAL(KIND=CF_DOUBLE) :: c50=0.64278760968653933d0
      REAL(KIND=CF_DOUBLE) :: s50=0.76604444311897804d0
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Ixyz,Ixyzwhite,Kxyz,Kxyzwhite,Labs,LCh, &
           Lab99d,LCh99d
!   X modification has to be applied to Whitepoint coordinates, too.
!   Otherwise, colorfulness C99d won't be zero.
      Ix=Ixyzwhite(1);Iy=Ixyzwhite(2);Iz=Ixyzwhite(3)
      Kxyzwhite=(/1.12d0*Ix-0.12d0*Iz,Iy,Iz/)
      Ix=Ixyz(1);Iy=Ixyz(2);Iz=Ixyz(3)
      Kxyz=(/1.12d0*Ix-0.12d0*Iz,Iy,Iz/)
      CALL xyz2lab(Kxyz,Kxyzwhite,Labs)
      Ls=Labs(1);as=Labs(2);bs=Labs(3)
!      L99d=325.22d0*log(1.d0+3.6d-3*Ls)
      L99d=scal_L99d*log(1.d0+3.6d-3*Ls)!exact normalisation to L=100
      e=as*c50+bs*s50
      f=1.14d0*(-as*s50+bs*c50)
!      WRITE(*,'(A,4(F16.8))') '**** a,b,e,f       =',as,bs,e,f
      G=sqrt(e*e+f*f)
      C99d=22.5d0*log(1.d0+0.06d0*G)
!---- Corrected arctan formula
!      IF (e>0.d0.AND.f>=0.d0) THEN
!         hef=atan(f/e)
!      ELSE IF (e==0.d0.AND.f>0.d0) THEN
!         hef=pi2
!      ELSE IF (e<0.d0) THEN
!         hef=pi+atan(f/e)
!      ELSE IF (e==0.d0.AND.f<0.d0) THEN
!         hef=1.5d0*pi
!      ELSE IF (e>0.d0.AND.f<=0.d0) THEN
!         hef=tau+atan(f/e)
!      ELSE!e==0 && f==0
!         hef=0.d0
!      ENDIF
      hef=atan2(f,e)!atan2 does quadrant checking
!      WRITE(*,'(A,4(F16.8))') '**** e,f,atan(f/e) =',e,f,hef
      h99d=hef+d50
      IF (h99d<0.d0) THEN
         h99d=h99d+tau
      ELSE IF (h99d>tau) THEN
         h99d=h99d-tau
      ENDIF
      a99d=C99d*cos(h99d)
      b99d=C99d*sin(h99d)
      Lab99d=(/L99d,a99d,b99d/)
      IF (degmode>=1) THEN
         LCh99d=(/L99d,C99d,h99d*r2d/)
      ELSE
         LCh99d=(/L99d,C99d,h99d/)
      ENDIF
      RETURN
    END SUBROUTINE XYZ2DIN99d

    SUBROUTINE LCh99d2XYZ(LCh99d,Ixyzwhite,Ixyz,degmode)
      IMPLICIT NONE
      INTEGER :: degmode
      REAL(KIND=CF_DOUBLE) :: scal_L99d=325.2194339489675351d0
      REAL(KIND=CF_DOUBLE) :: Ix,Iy,Iz,Kx,Ky,Kz,Ls,as,bs,L99d,C99d,h99d,a99d,b99d
      REAL(KIND=CF_DOUBLE) :: e,f,G,hef
      REAL(KIND=CF_DOUBLE) :: d50=0.87266462599716479d0
      REAL(KIND=CF_DOUBLE) :: c50=0.64278760968653933d0
      REAL(KIND=CF_DOUBLE) :: s50=0.76604444311897804d0
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Ixyz,Ixyzwhite,Kxyz,Kxyzwhite,Labs,LCh,Lab99d,LCh99d
      Ix=Ixyzwhite(1);Iy=Ixyzwhite(2);Iz=Ixyzwhite(3)
      Kxyzwhite=(/1.12d0*Ix-0.12d0*Iz,Iy,Iz/)
      L99d=LCh99d(1)
      C99d=LCh99d(2)
      h99d=LCh99d(3)
      IF (degmode>=1) THEN
         h99d=LCh99d(3)/r2d
      ELSE
         h99d=LCh99d(3)
      ENDIF
      G=(exp(C99d/22.5d0)-1.d0)/6.d-2
      e=G*cos(h99d-d50)
      f=G*sin(h99d-d50)
      as=e*c50-f/1.14d0*s50
      bs=e*s50+f/1.14d0*c50
!      Ls=(exp(L99d/325.22d0)-1.d0)/3.6d-3
      Ls=(exp(L99d/scal_L99d)-1.d0)/3.6d-3!exact normalisation to L=100
      Labs=(/Ls,as,bs/)
      CALL lab2xyz(Labs,Kxyz,Kxyzwhite)
      Kx=Kxyz(1);Ky=Kxyz(2);Kz=Kxyz(3)
      Iy=Ky;Iz=Kz
      Ix=(Kx+0.12d0*Kz)/1.12d0
      Ixyz=(/Ix,Iy,Iz/)
      RETURN
    END SUBROUTINE LCh99d2XYZ

    SUBROUTINE Lab99d2XYZ(Lab99d,Ixyzwhite,Ixyz)
      IMPLICIT NONE
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Ixyz,Ixyzwhite,Lab99d,LCh99d
      REAL(KIND=CF_DOUBLE) :: L99d,C99d,h99d,a99d,b99d,xdum
      L99d=Lab99d(1);a99d=Lab99d(2);b99d=Lab99d(3);
      C99d=sqrt(a99d**2+b99d**2)
      IF (C99d.gt.0.d0) THEN
         xdum=max(min(a99d/(C99d),1.d0),-1.d0)
      ELSE
         xdum=0.d0
      ENDIF
      h99d=acos(xdum)*sign(1.d0,b99d)
      LCh99d=(/L99d,C99d,h99d/)
      CALL LCh99d2XYZ(LCh99d,Ixyzwhite,Ixyz,0)
      RETURN
    END SUBROUTINE Lab99d2XYZ

    SUBROUTINE Lab2L99o(Lab,Lab99o,LCh99o,degmode)
      IMPLICIT NONE
      INTEGER :: degmode
      REAL(KIND=CF_DOUBLE) :: scal_L99o=303.6710054705098647d0
      REAL(KIND=CF_DOUBLE) :: d26=0.45378560551852569d0
      REAL(KIND=CF_DOUBLE) :: c26=0.89879404629916699d0
      REAL(KIND=CF_DOUBLE) :: s26=0.43837114678907742d0
      REAL(KIND=CF_DOUBLE) :: Ls,as,bs,L99o,a99o,b99o,C99o,h99o,eo,fo,G,hef
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Lab,Lab99o,LCh99o
      Ls=Lab(1); as=Lab(2); bs=Lab(3)
      L99o=scal_L99o/kE_DIN99*log(1.d0+3.9d-3*Ls)
      eo=as*c26+bs*s26
      fo=0.83d0*(-as*s26+bs*c26)
      G=sqrt(eo*eo+fo*fo)
      C99o=log(1.d0+0.075d0*G)/(0.0435d0*kE_DIN99*kCH_DIN99)
      hef=atan2(fo,eo)
      h99o=hef+d26
      IF (h99o<0.d0) THEN
         h99o=h99o+tau
      ELSE IF (h99o>tau) THEN
         h99o=h99o-tau
      ENDIF
      a99o=C99o*cos(h99o)
      b99o=C99o*sin(h99o)
      IF (degmode>=1) THEN
         LCh99o=(/L99o,C99o,h99o*r2d/)
      ELSE
         LCh99o=(/L99o,C99o,h99o/)
      ENDIF
      Lab99o=(/L99o,a99o,b99o/)
      RETURN
    END SUBROUTINE Lab2L99o

    SUBROUTINE LCh99o2Lab(LCh99o,Lab,degmode)
      IMPLICIT NONE
      INTEGER :: degmode
      REAL(KIND=CF_DOUBLE) :: scal_L99o=303.6710054705098647d0
      REAL(KIND=CF_DOUBLE) :: d26=0.45378560551852569d0
      REAL(KIND=CF_DOUBLE) :: c26=0.89879404629916699d0
      REAL(KIND=CF_DOUBLE) :: s26=0.43837114678907742d0
      REAL(KIND=CF_DOUBLE) :: Ls,as,bs,L99o,a99o,b99o,C99o,h99o,eo,fo,G,hef
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Lab,Lab99o,LCh99o
      L99o=LCh99o(1)
      C99o=LCh99o(2)
      IF (degmode>=1) THEN
         h99o=LCh99o(3)/r2d
      ELSE
         h99o=LCh99o(3)
      ENDIF
      Ls=(exp(kE_DIN99*L99o/scal_L99o)-1.d0)/3.9d-3
      hef=h99o-d26
      G=(exp(0.0435d0*kE_DIN99*kCH_DIN99*C99o)-1.d0)/0.075d0
      eo=G*cos(hef)
      fo=G*sin(hef)
      as=(eo*c26-(fo/0.83d0)*s26)
      bs=(eo*s26+(fo/0.83d0)*c26)
      Lab=(/Ls,as,bs/)
      RETURN
    END SUBROUTINE LCh99o2Lab

    SUBROUTINE Lab99o2Lab(Lab99o,Lab,degmode)
      IMPLICIT NONE
      INTEGER :: degmode
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: Lab99o,Lab,LCh99o
      CALL lab2lch(Lab99o,LCh99o,1,degmode)
      CALL LCh99o2Lab(LCh99o,Lab,degmode)
      RETURN
    END SUBROUTINE Lab99o2Lab

    SUBROUTINE xyz2ipt100(IXYZ_s,IXYZwhite_s,flgwhite,IPT100)
      IMPLICIT NONE
      INTEGER :: flgwhite
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_s,IXYZwhite_s,IPT100,IPT
      CALL xyz2ipt(IXYZ_s,IXYZwhite_s,flgwhite,IPT)
      IPT100(:)=IPT(:)*1.d2!scale top Lab-like range
      RETURN
    END SUBROUTINE xyz2ipt100

    SUBROUTINE ipt2ich(IPTio,IChio,flgdeg100io,direction)
      IMPLICIT NONE
      INTEGER :: flgdeg100io,direction, flgdeg100i,flgdeg100o
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IPTio,IChio,IPT,ICh
      REAL(KIND=CF_DOUBLE) :: I1,P1,T1
      REAL(KIND=CF_DOUBLE) :: CPT,hPT
      flgdeg100i=flgdeg100io/10
      flgdeg100o=flgdeg100io-10*flgdeg100i
!      write(*,*) 'flgdeg100io,flgdeg100i,flgdeg100o =',flgdeg100io,flgdeg100i,flgdeg100o
      IF (direction>=1) THEN!L*a*b* to L*C*h*
         IF (flgdeg100i>=1) THEN!input is IPT100
            IPT=1.d-2*IPTio
         ELSE
            IPT=IPTio
         ENDIF
         CPT=sqrt(IPT(2)**2+IPT(3)**2)
         hPT=atan2(IPT(3),IPT(2))
         IF (hPT<0.d0) THEN
            hPT=hPT+tau
         ELSE IF (hPT>tau) THEN
            hPT=hPT-tau
         ENDIF
         IF (flgdeg100o==2) THEN
            ICh=(/1.d2*IPT(1),1.d2*CPT,r2d*hPT/)!times 100 and degrees
         ELSE IF (flgdeg100o==1) THEN
            ICh=(/1.d2*IPT(1),1.d2*CPT,hPT/)!times 100
         ELSE
            ICh=(/IPT(1),CPT,hPT/)
         ENDIF
         IChio=ICh
      ELSE
         IF (flgdeg100i==2) THEN!input is ICh100deg
            ICh=(/1.d-2*IChio(1),1.d-2*IChio(2),d2r*IChio(3)/)
         ELSE IF (flgdeg100i==1) THEN!input is ICh100
            ICh=(/1.d-2*IChio(1),1.d-2*IChio(2),IChio(3)/)
         ELSE
            ICh=IChio
         ENDIF
!         CPT=sqrt(IPT(2)**2+IPT(3)**2)
!         hPT=atan2(IPT(3),IPT(2))
         I1=ICh(1)
         P1=ICh(2)*cos(ICh(3))
         T1=ICh(2)*sin(ICh(3))
         IF (flgdeg100o>=1) THEN
            IPT=1.d2*(/I1,P1,T1/)
         ELSE
            IPT=(/I1,P1,T1/)
         ENDIF
         IPTio=IPT
      ENDIF
      RETURN
    END SUBROUTINE ipt2ich

    SUBROUTINE xyz2ipt(IXYZ_s,IXYZwhite_s,flgwhite,IPT)
      USE chromadapt_matrices
      IMPLICIT NONE
      INTEGER :: flgwhite,selmatrix=3,i,j,k!CAT02
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_xyz2lms=RESHAPE((/0.4002d0, 0.7075d0,-0.0807d0, &
                                                                  -0.2280d0, 1.1500d0, 0.0612d0, &
                                                                   0.d0,     0.d0,     0.9184d0/),(/3,3/))
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_lms2ipt=RESHAPE((/0.4000d0, 0.4000d0, 0.2000d0, &
                                                                   4.4550d0,-4.8510d0, 0.3960d0, &
                                                                   0.8056d0, 0.3572d0,-1.1628d0/),(/3,3/))
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_s,IXYZ,IXYZwhite_s,LMS0,LMS1,IPT
      REAL(KIND=CF_DOUBLE) :: L0,M0,S0,L1,M1,S1, a=4.d0,b=3.d0,c=3.d0, beta_c=0.071d0,beta_h=0.03d0
      REAL(KIND=CF_DOUBLE) :: CPT,hPT
!      IXYZ=IXYZ_s
      IF (flgwhite>=1) THEN
         CALL chromadapt_xyz(IXYZ_s,IXYZwhite_s,whitepoint_d65, IXYZ, selmatrix)
      ELSE
         IXYZ=IXYZ_s
      ENDIF
      CALL operate_matrix3x3(mat_xyz2lms,IXYZ,LMS0)
      L0=LMS0(1); M0=LMS0(2); S0=LMS0(3)
!      L1=sign(abs(L0)**0.43d0,L0)
!      M1=sign(abs(M0)**0.43d0,M0)
!      S1=sign(abs(S0)**0.43d0,S0)
      CALL ipt_power43(L0,L1,1)
      CALL ipt_power43(M0,M1,1)
      CALL ipt_power43(S0,S1,1)
      LMS1=(/L1,M1,S1/)
      CALL operate_matrix3x3(mat_lms2ipt,LMS1,IPT)
!Original IPT correctly reproduces the sample values.
!export
      CPT=sqrt(IPT(2)**2+IPT(3)**2)
      hPT=atan2(IPT(3),IPT(2))
      IF (cf_verbose>=1) THEN
!         WRITE(*,*) '**** IXYZ_s=',IXYZ_s
!         WRITE(*,*) '**** IXYZ  =',IXYZ
!         WRITE(*,*) '**** xyz_s =',IXYZ_s/sum(IXYZ_s)
      ENDIF
      RETURN
    END SUBROUTINE xyz2ipt

    SUBROUTINE ipt2xyz(IPT_i,IXYZ_d,IXYZwhite_d,flgwhite,flg100)
      USE chromadapt_matrices
      IMPLICIT NONE
      INTEGER :: flgwhite,flg100,selmatrix=3,i,j,k!CAT02
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_xyz2lms=RESHAPE((/0.4002d0, 0.7075d0,-0.0807d0, &
                                                                  -0.2280d0, 1.1500d0, 0.0612d0, &
                                                                   0.d0,     0.d0,     0.9184d0/),(/3,3/))
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_lms2ipt=RESHAPE((/0.4000d0, 0.4000d0, 0.2000d0, &
                                                                   4.4550d0,-4.8510d0, 0.3960d0, &
                                                                   0.8056d0, 0.3572d0,-1.1628d0/),(/3,3/))
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_ipt2lms,mat_lms2xyz
      REAL(KIND=CF_DOUBLE) :: det_xyz2lms,det_lms2ipt
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IPT_i,IXYZ_d,IXYZ,IXYZwhite_d,LMS0,LMS1,IPT
      REAL(KIND=CF_DOUBLE) :: L0,M0,S0,L1,M1,S1, a=4.d0,b=3.d0,c=3.d0, beta_c=0.071d0,beta_h=0.03d0
      REAL(KIND=CF_DOUBLE) :: CPT,hPT
      LOGICAL :: setup_iptbwd=.TRUE.
      SAVE setup_iptbwd,mat_lms2xyz,mat_ipt2lms
      IF (setup_iptbwd) THEN
         CALL invert_matrix3x3(mat_xyz2lms,mat_lms2xyz,det_xyz2lms)
         CALL invert_matrix3x3(mat_lms2ipt,mat_ipt2lms,det_lms2ipt)
         setup_iptbwd=.FALSE.
      ENDIF
      IF (flg100>=1) THEN
         IPT=1.d-2*IPT_i
      ELSE
         IPT=IPT_i
      ENDIF
      CALL operate_matrix3x3(mat_ipt2lms,IPT,LMS1)
      L1=LMS1(1); M1=LMS1(2); S1=LMS1(3)
!      L0=sign(abs(L1)**(1.d0/0.43d0),L1)
!      M0=sign(abs(M1)**(1.d0/0.43d0),M1)
!      S0=sign(abs(S1)**(1.d0/0.43d0),S1)
      CALL ipt_power43(L1,L0,-1)
      CALL ipt_power43(M1,M0,-1)
      CALL ipt_power43(S1,S0,-1)
      LMS0=(/L0,M0,S0/)
      CALL operate_matrix3x3(mat_lms2xyz,LMS0,IXYZ)
      IF (flgwhite>=1) THEN
         CALL chromadapt_xyz(IXYZ,whitepoint_d65,IXYZwhite_d, IXYZ_d, selmatrix)
      ELSE
         IXYZ_d=IXYZ_d
      ENDIF
      IF (cf_verbose>=1) THEN
!         WRITE(*,*) '**** IPT   =',IPT
!         WRITE(*,*) '**** LMS1  =',LMS1
!         WRITE(*,*) '**** LMS0  =',LMS0
!         WRITE(*,*) '**** IXYZ  =',IXYZ
!         WRITE(*,*) '**** IXYZ_d=',IXYZ_d
!         WRITE(*,*) '**** xyz =',IXYZ/sum(IXYZ)
!         WRITE(*,*) '**** xyz_d =',IXYZ_d/sum(IXYZ_d)
      ENDIF
      RETURN
    END SUBROUTINE ipt2xyz

    SUBROUTINE xyz2ipteuc(IXYZ_s,IXYZwhite_s,flgwhite,IPTE)
! See Shizhe Shen thesis on "Color Difference Formula and Uniform Color Space Modeling and Evaluation"
      USE chromadapt_matrices
      IMPLICIT NONE
      INTEGER :: flgwhite,selmatrix=3,i,j,k!CAT02
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_com,mat_temp
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_s,IXYZ,IXYZwhite_s,LMS0,LMS1,IPT0,IPT1,IPTE,IPT
      REAL(KIND=CF_DOUBLE) :: L0,M0,S0,L1,M1,S1, a=4.d0,b=3.d0,c=3.d0, beta_c=0.071d0,beta_h=0.03d0
      REAL(KIND=CF_DOUBLE) :: CPT,hPT
      REAL(KIND=CF_DOUBLE) :: Itmp,I1,P1,T1,CPTE,hPTE,CPT1,hPT1,IE,PE,TE
      LOGICAL :: setup_ipt=.TRUE.
      SAVE setup_ipt,mat_com
      IF (setup_ipt) THEN
! NOTE: Shizhe Shen improvement fits test results with beta_c=0.071 instead of 0.06
!       (see section 5.6.2 and table 5.7).
!       Transposition is NOT to be used (comment line left in for documentary purpose only).
!       I.e. the apparent transposition of the COM coeffs is likely a typo.
         CALL product_matrix3x3(euc_mat_red,euc_mat_rot,mat_temp)
         CALL product_matrix3x3(mat_temp,euc_mat_scal,mat_com)
!         CALL transpose_matrix3x3(mat_com,mat_temp); mat_com=mat_temp! Transposition appears to be wrong
         IF (cf_verbose>=1) THEN
            WRITE(*,'(A)') '        M_COM:'
            WRITE(*,'(A,3(1X,F16.12))') 'a11,a12,a13 =',(mat_com(1,j),j=1,3)
            WRITE(*,'(A,3(1X,F16.12))') 'a21,a22,a23 =',(mat_com(2,j),j=1,3)
            WRITE(*,'(A,3(1X,F16.12))') 'a31,a32,a33 =',(mat_com(3,j),j=1,3)
         ENDIF
         setup_ipt=.FALSE.
      ENDIF
!      IXYZ=IXYZ_s
      CALL xyz2ipt(IXYZ_s,IXYZwhite_s,flgwhite,IPT0)
      CALL operate_matrix3x3(mat_com,IPT0,IPT1)
      CPT1=sqrt(IPT1(2)**2+IPT1(3)**2)
      hPT1=atan2(IPT1(3),IPT1(2))
      I1=IPT1(1);P1=IPT1(2); T1=IPT1(3)
      Itmp=I1/1.d2
      IE=1.d2*(c*Itmp**(a-1.d0) * (1.d0-Itmp)**(b-1.d0) + Itmp)
      CPTE=log(1.d0+beta_c*CPT1)/beta_c
      hPTE=hPT1
      PE=P1*CPTE/max(CPT1,1.d-30)
      TE=T1*CPTE/max(CPT1,1.d-30)
!export
      IPTE=(/IE,PE,TE/)
      IF (cf_verbose>=1) THEN
!         WRITE(*,*) '**** IXYZ_s=',IXYZ_s
!         WRITE(*,*) '**** xyz_s =',IXYZ_s/sum(IXYZ_s)
!         WRITE(*,*) '**** IXYZ_w=',IXYZwhite_s
!         WRITE(*,*) '**** IXYZ  =',IXYZ
!         WRITE(*,*) '**** IPT0  =',IPT0
!         WRITE(*,*) '**** IPT1  =',IPT1
!         WRITE(*,*) '**** I1,CPT1,hPT1=',I1,CPt1,hPT1,'**** IE,CPTE,hPTE=',IE,CPTE,hPTE
!         WRITE(*,*) '**** I1,P1,T1=',I1,P1,T1
!         WRITE(*,*) '**** IE,PE,TE=',IE,PE,TE
!         WRITE(*,*) '**** IPTE=',IPTE
!         WRITE(*,*) '**** IPT =',IPT
      ENDIF
      RETURN
    END SUBROUTINE xyz2ipteuc

    SUBROUTINE ipteuc2xyz(IPTE,IXYZ_d,IXYZwhite_d,flgwhite)
      USE chromadapt_matrices
      IMPLICIT NONE
      INTEGER :: flgwhite,selmatrix=3,i,j,k,iter!CAT02
      REAL(KIND=CF_DOUBLE),DIMENSION(3,3) :: mat_com,mat_moc,mat_temp
      REAL(KIND=CF_DOUBLE),DIMENSION(3) :: IXYZ_D,IXYZ,IXYZwhite_D,LMS0,LMS1,IPT0,IPT1,IPTE,IPT
      REAL(KIND=CF_DOUBLE) :: L0,M0,S0,L1,M1,S1, a=4.d0,b=3.d0,c=3.d0, beta_c=0.071d0,beta_h=0.03d0
      REAL(KIND=CF_DOUBLE) :: CPT,hPT,det_com,xie,xi1
      REAL(KIND=CF_DOUBLE) :: Itmp,I1,P1,T1,CPTE,hPTE,CPT1,hPT1,IE,PE,TE
      LOGICAL :: setup_ipt=.TRUE.
      SAVE setup_ipt,mat_com,mat_moc
      IF (setup_ipt) THEN
! NOTE: Shizhe Shen improvement fits test results with beta_c=0.071 instead of 0.06
!       (see section 5.6.2 and table 5.7).
!       Transposition is NOT to be used (comment line left in for documentary purpose only).
!       I.e. the apparent transposition of the COM coeffs is likely a typo.
         CALL product_matrix3x3(euc_mat_red,euc_mat_rot,mat_temp)
         CALL product_matrix3x3(mat_temp,euc_mat_scal,mat_com)
         CALL invert_matrix3x3(mat_com,mat_moc,det_com)
         IF (cf_verbose>=1) THEN
            WRITE(*,'(A)') '        M_COM:'
            WRITE(*,'(A,3(1X,F16.12))') 'a11,a12,a13 =',(mat_com(1,j),j=1,3)
            WRITE(*,'(A,3(1X,F16.12))') 'a21,a22,a23 =',(mat_com(2,j),j=1,3)
            WRITE(*,'(A,3(1X,F16.12))') 'a31,a32,a33 =',(mat_com(3,j),j=1,3)
         ENDIF
         setup_ipt=.FALSE.
      ENDIF
      IE=IPTE(1); PE=IPTE(2); TE=IPTE(3)
      CPTE=sqrt(PE**2+TE**2)!sqrt(P1^2*(CE/C1)**2+T1^2*(CE/C1)**2)=sqrt(P1^2+T1^2)*(CE/C1)=C1*CE/C1
      hPTE=atan2(TE,PE)
      hPT1=hPTE
      CPT1=(exp(beta_c*CPTE)-1.d0)/beta_c
      P1=PE*CPT1/max(CPTE,1.d-30)
      T1=TE*CPT1/max(CPTE,1.d-30)
      xie=1.d-2*IE
      CALL goalseek_ie2i1(xie,xi1,a,b,c,iter)
      I1=1.d2*xi1
      IPT1=(/I1,P1,T1/)
      CALL operate_matrix3x3(mat_moc,IPT1,IPT0)
      CALL ipt2xyz(IPT0,IXYZ_d,IXYZwhite_d,flgwhite,0)
      IF (cf_verbose>=1) THEN
!         WRITE(*,*) '**** IXYZ_s=',IXYZ_s
!         WRITE(*,*) '**** xyz_s =',IXYZ_s/sum(IXYZ_s)
!         WRITE(*,*) '**** IXYZ_w=',IXYZwhite_s
         WRITE(*,*) '**** IE,CPTE,hPTE=',IE,CPTE,hPTE,' **** I1,CPT1,hPT1=',I1,CPt1,hPT1,iter
         WRITE(*,*) '**** IE,PE,TE=',IE,PE,TE,' **** I1,P1,T1=',I1,P1,T1,iter
         WRITE(*,*) '**** I1,P1,T1=',I1,P1,T1
         WRITE(*,*) '**** IPTE =',IPTE
         WRITE(*,*) '**** IPT1 =',IPT1
         WRITE(*,*) '**** IPT0 =',IPT0
         WRITE(*,*) '**** IXYZd=',IXYZ_d
!         stop
      ENDIF
      RETURN
    END SUBROUTINE ipteuc2xyz

    SUBROUTINE goalseek_ie2i1(xie_in,xi1_out,a,b,c,iter)
      IMPLICIT NONE
      INTEGER :: i,iter,maxiter=100
      REAL(KIND=CF_DOUBLE) :: xie_in,xi1_out,a,b,c
      REAL(KIND=CF_DOUBLE) :: xprobe,ygoal,x1=-0.5d0,x2=1.5d0
      REAL(KIND=CF_DOUBLE) :: f_ie,df_ie,delta_f,delta_x,xacc=1.d-14
      !  xprobe=0.5d0*(x1+x2)
      xprobe=xie_in
      ygoal=xie_in
      iter=0
      DO i=0,maxiter
         iter=iter+1
!         CALL f2solve(xprobe,f,dfdx)
         f_ie=c*xprobe**(a-1.d0) * (1-xprobe)**(b-1.d0)+xprobe
         df_ie=c*((a-1.d0)*xprobe**(a-2.d0) * (1-xprobe)**(b-1.d0) &
              - xprobe**(a-1.d0) * (b-1.d0)*(1-xprobe)**(b-2.d0))  &
              + 1.d0
         delta_f=f_ie-ygoal
         delta_x=delta_f/df_ie
         xprobe=xprobe-delta_x
         IF((x1-xprobe)*(xprobe-x2).lt.0.d0) then
!            WRITE(*,*) 'Newton sequence diverged'
            write(*,*) 'x,f,df =',xprobe,f_ie,df_ie
            STOP
         ENDIF
         IF (abs(delta_x).lt.xacc) then
            xi1_out=xprobe
            RETURN
         ENDIF
      ENDDO
      WRITE(*,*) 'Newton exceeded maximum iterations'
    END SUBROUTINE goalseek_ie2i1

    SUBROUTINE ipt_power43(xi,xo,direction)
      IMPLICIT NONE
      INTEGER :: direction
      DOUBLE PRECISION :: xi,xo
      IF (direction>=1) THEN
         IF (xi==0.d0) THEN
            xo=0.d0
         ELSE
            xo=sign(abs(xi)**0.43d0,xi)
         ENDIF
      ELSE
         IF (xi==0.d0) THEN
            xo=0.d0
         ELSE
            xo=sign(abs(xi)**(1.d0/0.43d0),xi)
         ENDIF
      ENDIF
      RETURN
    END SUBROUTINE ipt_power43
    
! To be added: IPT2XYZ
! a=4, b=3, c=3
! E100=c*Itmp^(a-1) * (1-Itmp)^(b-1) + Itmp
! E100=3*Itmp^3 * (1-Itmp)^2 + Itmp = 3*Itmp * (1-2*Itmp+Itmp^2) + Itmp
!     =3*Itmp^3 - 6*Itmp^4 + 3*Itmp^5 + Itmp
!     =3*(Itmp^3 - 2*Itmp^4 + Itmp^5) + Itmp

END MODULE color_functions_extended
