      SUBROUTINE DAT2JD(yr,mo,dd,hh,mm,ss,mode,JD)
!     *** Calculation of Julian Date by Meeus "Astronomical Algorithms" ***
!           -- Input --
!     yr,mo,dd : Year, Month, Day (integers)
!     hh,mm,ss : Hours, Minutes, Seconds UTC (integers)
!     mode     : Julian Date modifier:
!                0 - default JD
!                1 - modified JD (MJD) = days since Nov. 17 1858 = JD-2400000.5
!                2 - 2nd MJD           = days since May. 24 1968 = JD-2440000.5
!                3 - Y0                = days since Jan. 01 0000 = JD-1721057.5
!                4 - Y2K               = days since Jan. 01 2000 = JD-2451544.5
!                5 - F77               = days since Jan. 01 1970 = JD-2440587.5
!           -- Output --
!     JD       : Julian Date (double precision)
      IMPLICIT NONE
      INTEGER i,j,mode,yr,mo,dd,hh,mm,ss,Y,M,A,B,Zoff
      DOUBLE PRECISION JD,JD0,jdoff,F,yroff
      PARAMETER (Zoff=32142)
      INTEGER mdays(12,0:2),feb29
      LOGICAL isleap
      DATA ((mdays(i,j),i=1,12),j=0,1)/                                 &
     &31,28,31,30,31,30,31,31,30,31,30,31,                              &
     &31,29,31,30,31,30,31,31,30,31,30,31/
      yroff = Zoff/365.25d0
!---  Check if input values are within the defined range
      IF (isleap(yr)) THEN
         feb29 = 1
      ELSE
         feb29 = 0
      ENDIF
      IF (mo.lt.1 .OR. mo.gt.12) THEN
         WRITE(*,*) 'Month number out of range!',mo
         GOTO 66
      ELSE IF (dd.lt.1 .OR. dd.gt.mdays(mo,feb29)) THEN
         WRITE(*,*) 'Day number out of range!',dd
         GOTO 66
      ELSE IF (yr.eq.1582.and.mo.eq.10.and.(dd.gt.4.and.dd.lt.15)) THEN
         WRITE(*,*)                                                     &
     &   'Date does not exist due to Gregorian calendar reformation.'
         GOTO 66
      ENDIF
!---  Calendar formulae
      IF (mo.le.2) THEN
         Y = yr-1
         M = mo+12
      ELSE
         Y = yr
         M = mo
      ENDIF
      IF ((yr.eq.1582.AND.mo.eq.10.AND.dd.ge.15) .OR.                   &
     &    (yr.eq.1582.AND.mo.gt.10)              .OR.                   &
     &    (yr.gt.1582))                               THEN
         A = int(Y/100)
         B = 2-A+int(A/4)
      ELSE
         B = 0
      ENDIF
      JD0 = int(365.25d0*(Y+4716+yroff))+int(30.6001*(M+1))+dd+B-1524
      CALL MJDOFF(mode,1,jdoff)
      F = (hh*3.6d3+mm*6.d1+ss)/8.64d4
      JD = JD0-jdoff+F-Zoff
      RETURN
   66 WRITE(*,*) '--> Abort'
      STOP
      END

      SUBROUTINE JD2DAT(JD,yr,mo,dd,hh,mm,ss,mode)
!     Calculation of civil date from Julian date by Meeus "Astronomical Algorithms"
!           -- Input --
!     JD       : Julian Date (double precision)
!     mode     : Julian date modifier (see at top)
!           -- Output --
!     yr,mo,dd : Year, Month, Day (integers)
!     hh,mm,ss : Hours, Minutes, Seconds UTC (integers)
      IMPLICIT NONE
      INTEGER mode,Z,Z0,A,B,C,D,E,alpha,yr,mo,dd,hh,mm,ss,ssday
      INTEGER Zoff,yroff
      DOUBLE PRECISION JD,JD0,F,JD0G,jdoff
!      PARAMETER (Zoff=0)
      PARAMETER (Zoff=32142)
      PARAMETER (Z0=2299161+Zoff,JD0G=1867216.25d0+Zoff)
      yroff = Zoff/365.25d0
      CALL MJDOFF(mode,1,jdoff)
      JD0 = JD+jdoff+Zoff
      Z = int(JD0)
      F = JD0-Z
      IF (Z .lt. Z0) THEN
!--   Datum vor der Gregorianischen Kalenderreform 15.10.1582
         A = Z
      ELSE
         alpha = int((Z-JD0G)/36524.25d0)
         A = Z+1+alpha-alpha/4 ! alpha/4 = int(alpha/4), da alpha=integer
      ENDIF
      B = A+1524
      C = int((B-122.1d0)/365.25d0)
      D = int(365.25d0*C)
      E = int((B-d)/30.6001d0)
!--   Monatstag
      dd = B-D-int(30.6001d0*E)
!--   Monatszahl
      IF (E.lt.14) THEN
         mo = E-1
      ELSE !IF (E.eq.14 .OR. E.eq.15) THEN
         mo = E-13
      ENDIF
!--   Jahreszahl
      IF (mo.gt.2) THEN
         yr = C-4716-yroff
      ELSE
         yr = C-4715-yroff
      ENDIF
!      hh = int(24.d0*F)
!      mm = int(1.44d3*F)-60*hh
!      ss = int(8.64d4*F)-3600*hh-60*mm
! alternative method
      ssday=nint(8.64d4*F)
      hh = ssday/3600
      mm = ssday/60-60*hh
      ss = ssday-60*mm-3600*hh
!      CALL MD2DOY(yr,mo,dd,doy)
      RETURN
      END

!===== Useful Functions ================================================
      SUBROUTINE MD2DOY(yr,mo,dd,doy)
!     Returns the day of the year from 0 (1.1.) to 364/365 (31.12.)
!     yr,mo,dd : Year, Month, Day      (Input)
!     doy      : _D_ay _O_f the _Y_ear (Output)
      IMPLICIT NONE
      INTEGER i,yr,mo,dd,sd1582,mdays(12),doy
      LOGICAL isleap,sj
      DATA mdays /31,28,31,30,31,30,31,31,30,31,30,31/
      IF (yr.eq.1582.and.mo.eq.10.and.(dd.gt.4.and.dd.lt.15)) THEN
         WRITE(*,*)                                                     &
     &   'Date does not exist due to Gregorian calendar reformation.'
         STOP
      ENDIF
      sj = isleap(yr)
      sd1582 = 0
      IF (mo.lt.1 .OR. mo.gt.12) GOTO 66
      IF (sj) THEN
         mdays(2) = 29
      ELSE
         mdays(2) = 28
      ENDIF
      IF (yr.eq.1582) THEN
         IF ((mo.eq.10.and.dd.ge.15).OR.(mo.gt.10)) THEN
            sd1582=-10
         ENDIF
      ENDIF
      IF (dd.lt.1 .OR. dd.gt.mdays(mo)) GOTO 77
      doy = dd+sd1582-1
      DO 10 i=1,mo-1
         doy = doy+mdays(i)
  10  ENDDO
      RETURN
  66  CONTINUE
      WRITE(*,*) 'Bad Month Number'
      STOP
  77  CONTINUE
      WRITE(*,*) 'Bad Day Number'
      END

      SUBROUTINE JD2DOW(JD,dow,cdow,mode,lang)
!---  Returns the Day of the Week of given Julian Day
!     -- Input --
!     JD   : Julian Days since 1.1.-4712 12 UT
!     mode : Julian Date modifier: 0 - default JD
!                                  1 - modified JD (MJD) = JD-2400000.5
!                                  2 - 2nd MJD           = JD-2440000.5
!     lang : Language selection:   0 - english
!                                  1 - german
!     -- Output --
!     dow  : number of the day (0--6)
!     cdow : abbreviation (en: Sun--Sat / de: So--Sa)
      IMPLICIT NONE
      INTEGER IJD,dow,mode,lang,langsw
      DOUBLE PRECISION JD,jdoff
      CHARACTER*3 week(0:13),cdow
!      DATA week/'Mon','Tue','Wed','Thu','Fri','Sat','Sun',              &
!     &          'Mo ','Di ','Mi ','Do ','Fr ','Sa ','So '/
      DATA week/'Sun','Mon','Tue','Wed','Thu','Fri','Sat',              &
     &          'Su ','Mo ','Di ','Mi ','Do ','Fr ','Sa '/
      CALL MJDOFF(mode,1,jdoff)
      IJD = int(JD+32145+jdoff)
      dow = mod(IJD,7)
      IF (dow.lt.0) dow = dow+7
      langsw = 7*max(min(lang,1),0)
      cdow= week(dow+langsw)
      RETURN
      END

      SUBROUTINE MJDOFF(mode,md2,jdoff)
!--   Offset for Modified Julian Date and 12 UT correction for JD
      IMPLICIT NONE
      INTEGER mode,md2
      DOUBLE PRECISION jdoff
      IF (mode.eq.1) THEN
         jdoff = 2400000.5D0    ! MJD
      ELSE IF (mode.eq.2) THEN
         jdoff = 2440000.5D0    ! MMJD
      ELSE IF (mode.eq.3) THEN
         jdoff = 1721057.5D0    ! Y0
      ELSE IF (mode.eq.4) THEN
         jdoff = 2451544.5D0    ! Y2K
      ELSE IF (mode.eq.5) THEN
         jdoff = 2440587.5D0    ! F77 TIME() reference
      ELSE IF (mode.eq.6) THEN
         jdoff = 2440400.5D0    ! DE405 zero
      ELSE
         jdoff = 0.0D0
      ENDIF
      IF (md2 .ge. 1) jdoff = jdoff+0.5D0
      RETURN
      END

      LOGICAL FUNCTION isleap(j)
!--   .TRUE. if year is a leap-year
      IMPLICIT NONE
      INTEGER j
      LOGICAL q4,q100,q400,isgreg,teilt
      q4   = teilt(j,4)
      q100 = teilt(j,100)
      q400 = teilt(j,400)
      isgreg = j .ge. 1582
      IF (isgreg) THEN
         isleap = q4 .and. (.not. q100 .or. q400)
      ELSE
         isleap = q4
      ENDIF
      RETURN
      END

      LOGICAL FUNCTION teilt(n,k)
      IMPLICIT NONE
      INTEGER n,k
      teilt = mod(n,k) .eq. 0
      RETURN
      END
