MODULE sunpath_core
  REAL(KIND=8) :: JDnow,latitude,longitude,z_surface=0.d0,pscal_surface=1.d0,pscal_elev=1.d0
  REAL(KIND=8) :: RA_deg,DC_deg,azimuth,altitude,alt_refract,corr_refract
  REAL(KIND=8) :: eot_min,r_deg,th_mean_local
END MODULE sunpath_core

MODULE sunpath_constants
  IMPLICIT NONE
  REAL(KIND=8) :: pi= 3.14159265358979323846d0
  REAL(KIND=8) :: pi2=1.57079632679489661923d0
  REAL(KIND=8) :: ci= 6.28318530717958647693d0
  REAL(KIND=8) :: d2r=1.74532925199432957692d-2
  REAL(KIND=8) :: ecc=0.0167d0!used for original calculation
!  REAL(KIND=8) :: ecc=0.01671123d0!a bit more accurate
  REAL(KIND=8) :: JD2000=2451545.d0
  REAL(KIND=8) :: JCentury=365.25d2!Julian century
  REAL(KIND=8) :: zscal=8400.d0,kappa_extinct=0.237d0,kglob_extinct=0.156d0!using ASTMG173,direct
  REAL(KIND=8) :: kappa_uvskin=2.3395d0,kglob_uvskin=1.9555d0!using ASTMG173 and CIE UV action profile
!  REAL(KIND=8),DIMENSION(1:5) :: kappa_ubvri=(/0.6d0,0.4d0,0.2d0,0.1d0,0.08d0/)!some website
!  REAL(KIND=8),DIMENSION(1:5) :: kappa_ubvri=(/0.82d0,0.30d0,0.22d0,0.16d0,0.13d0/)!using WFI filter set and ASTMG173,direct
  REAL(KIND=8),DIMENSION(1:5) :: kappa_ubvri=(/0.822d0,0.304d0,0.219d0,0.165d0,0.133d0/)!using WFI filter set and ASTMG173,direct
  REAL(KIND=8),DIMENSION(1:5) :: kappa_ugriz=(/0.540d0,0.269d0,0.172d0,0.137d0,0.145d0/)!using MEGAPRIME filter set and ASTMG173,direct
  REAL(KIND=8),DIMENSION(1:5) :: kglob_ubvri=(/0.517d0,0.183d0,0.135d0,0.097d0,0.083d0/)!using WFI filter set and ASTMG173,direct
  REAL(KIND=8),DIMENSION(1:5) :: kglob_ugriz=(/0.321d0,0.163d0,0.102d0,0.078d0,0.097d0/)!using MEGAPRIME filter set and ASTMG173,direct
  REAL(KIND=8) :: illum_const=1.33d5
END MODULE sunpath_constants

SUBROUTINE calc_sunpath
! Calculate path of the Sun using formulae from
! http://de.wikipedia.org/wiki/Sonnenstand (access: 2010-04-05)
! INPUT:
! JDnow        = Julian Date (standard)
! latitude  = geogr. latitude in degrees
! longitude = geogr. longitude in degrees

! OUTPUT:
! RA,DC     = rectascension, declination in degrees
! azimuth   = azimuth position in degrees (nautical) (alt_refract with refraction)
! altitude  = altitude in degrees
! eot_min   = equation of time in minutes
  USE sunpath_core
  USE sunpath_constants
  IMPLICIT NONE
  INTEGER :: yr,mo,dd,hh,mm,ss
  REAL(KIND=8) :: JD0,t2000,L,g,alpha,delta,Lambda,etilt
  REAL(KIND=8) :: TD0,th,theta,theta_gh,theta_g,tau,eot
  REAL(KIND=8) :: ctau,stau,clat,slat,ceps,slam,clam,cdel,sdel,tdel
  JD0=int(JDnow+0.5d0)-0.5d0
  CALL calc_anomaly(JDnow,t2000,L,g)
  CALL apparent_sun(g,r_deg)
! Lambda 1. from ecc, and 2. approximation with ecc=0.0167
  Lambda=L+2.d0*ecc*sin(g)+5.d0/4.d0*ecc**2*sin(2.d0*g) ! 1.
!  Lambda=L+(1.915d0*sin(g)+0.02d0*sin(2.d0*g))*d2r      ! 2.
  etilt=(23.439d0-4.d-7*t2000)*d2r
  ceps=cos(etilt)
  slam=sin(Lambda)
  clam=cos(Lambda)
  alpha=atan(ceps*slam/clam)
  IF (clam<0.d0) alpha=alpha+pi
!  eot_min=(L-ci*int(L/ci)-alpha)*4.d0/d2r
  eot=(L-alpha)
  CALL adjust_angle(eot,eot,0)
  eot_min=eot*4.d0/d2r
  delta=asin(sin(etilt)*sin(Lambda))
! azimuth, altitude for Greenwich
  th  = 24.d0*(JDnow-JD0)!UT in h
  th_mean_local=th+longitude*4.d0
!  TD0 = (JD0-JD2000)/JCentury!please check JDnow vs. JD0!
!  theta_gh = 6.697376d0+2400.05134d0*TD0+1.002738d0*th
!  theta_gh = theta_gh-24.d0*int(theta_gh/24.d0)
!  theta_g  = theta_gh*15.d0!hours to degrees
!  theta    = theta_g+longitude
!  theta    = theta-3.6d2*int(theta/3.6d2)
  CALL sidereal_time(JDnow,longitude,theta)!more accurate formula
  tau      = theta*d2r-alpha
  ctau=cos(tau)
  stau=sin(tau)
  clat=cos(latitude*d2r)
  slat=sin(latitude*d2r)
  cdel=cos(delta)
  sdel=sin(delta)
  tdel=tan(delta)
  azimuth=atan(stau/(ctau*slat-tdel*clat))+pi! pi offset for ref=north
  IF (ctau*slat-tdel*clat<0.d0) azimuth=azimuth+pi! fix quadrant
  azimuth=(azimuth-ci*int(azimuth/ci)) / d2r
  altitude=asin(cdel*ctau*clat+sdel*slat) / d2r
  CALL refract_correction(altitude,alt_refract,corr_refract)
  RA_deg = alpha/d2r
  DC_deg = delta/d2r
!-- Test output
!  write(*,*) 'th,JDnow,JD0 =',th,JDnow,JD0
!  write(*,*) 'th,n,L,g =',th,t2000,L/d2r,g/d2r
!  write(*,*) 'th,n,L,alpha =',th,t2000,L/d2r,alpha/d2r
!  write(*,*) 'th,a,d,TD0 =',th,alpha/d2r,delta/d2r,TD0
!  write(*,*) 'th,thetagh* =',th,theta_gh,theta
!  write(*,*) 'th,n,L,alpha =',th,t2000,eot_min
END SUBROUTINE calc_sunpath

SUBROUTINE sidereal_time(JD,long_deg,theta_deg)
  USE sunpath_constants, ONLY : JD2000,JCentury
  IMPLICIT NONE
  REAL(KIND=8) JD,long_deg,theta_deg
  REAL(KIND=8) t,theta0
  t=(JD-JD2000)/JCentury
  theta0=280.46061837d0 + 13185000.77d0*t+t**2/2577.765d0-T**3/3.871d7
  theta_deg=theta0+long_deg
  theta_deg=theta_deg-3.6d2*int(theta_deg/3.6d2)!trim angle
END SUBROUTINE sidereal_time

SUBROUTINE refract_correction(h,hr,dh)
! Average refraction correction for 10 °C and 1010 hPa
! and approximate surface altitude correction
! h  = geometrical altitude in degrees
! hr = apparent altitude in degrees
  USE sunpath_core,      ONLY : z_surface
  USE sunpath_constants, ONLY : zscal,d2r
  IMPLICIT NONE
!  REAL(KIND=8),PARAMETER :: d2r=1.74532925199432957692d-2
  REAL(KIND=8) :: h,hr,hu,dh,a,kscal
  kscal=exp(-z_surface/zscal)
  hu=max(h,-0.85d0)
  a = (hu+10.3d0/(hu+5.11d0))*d2r
  dh = 1.7d-2*kscal/tan(a)
  hr= h + dh
END SUBROUTINE refract_correction

SUBROUTINE calc_airmass(h,x,eta,etag,etad)
  USE sunpath_core,      ONLY : z_surface,pscal_surface,pscal_elev
  USE sunpath_constants, ONLY : zscal,d2r
  IMPLICIT NONE
  INTEGER :: i
  REAL(KIND=8) :: ftwilight,fbaro
  REAL(KIND=8) :: h,x,eta(0:11),etag(0:11),etad(0:11),hr,dh,x0,a,kscal
  REAL(KIND=8) :: eta38(0:11),etag38(0:11),etad38(0:11),hrmin=-0.25d0,xmax=38.5d0
  REAL(KIND=8) :: ktwilight(0:11)=(/1.d0,0.5d0,0.9d0,1.1d0,1.4d0,2.d0, &
                                        0.5d0,1.d0,1.4d0,2.d0,2.5d0,0.5d0/)
!  kscal=exp(-z_surface/zscal)
  kscal=fbaro(z_surface)*pscal_surface
  pscal_elev=kscal
  CALL refract_correction(h,hr,dh)
  IF (hr<=hrmin) THEN
     x=230.d0
  ELSE
     a=(hr + 244.d0 / (165.d0 + 47.d0 * max(hr,1.d-99)**1.1d0)) * d2r
     x0=1.d0/sin(a)
     x=kscal*x0
  ENDIF
  CALL calc_transmission(x,eta,etag,etad)
  CALL calc_transmission(xmax,eta38,etag38,etad38)
  DO i=0,11
     etag(i)=(max(etad(i),etad38(i))+0.009d0)*ftwilight(h)**ktwilight(i)+eta(i)!use true altitude here
!        etag(i)=(etag38(i)-eta38(i))+eta(i)!testing
!        if(abs(hr)<0.1d0) write(*,*) 'ftwilight(',h,') = ',ftwilight(h)
  ENDDO
END SUBROUTINE calc_airmass

FUNCTION fbaro(z)
  IMPLICIT NONE
  REAL(KIND=8) :: fbaro,z,prel
  REAL(KIND=8) :: beta=-6.5d-3,gamma=5.255889d0,t0=288.15d0
  IF (z<=11.d3) THEN
     fbaro=(1.d0-6.5d-3*z/t0)**5.2558889d0
  ELSE
!     fbaro=0.223360290552d0/(z-11.d3)**6341.6046d0
     fbaro=0.223360290552d0*exp((11.d3-z)/6341.6046d0)
  ENDIF
  RETURN
END FUNCTION fbaro

SUBROUTINE calc_transmission(x,eta,etag,etad)
  USE sunpath_constants, ONLY : zscal,d2r,kappa_extinct,kappa_uvskin,kappa_ubvri,kappa_ugriz,&
                                          kglob_extinct,kglob_uvskin,kglob_ubvri,kglob_ugriz
  IMPLICIT NONE
  INTEGER :: i
  REAL(KIND=8) :: x,eta(0:11),etag(0:11),etad(0:11)
  eta(0)=exp(-kappa_extinct*x)
  eta(11)=exp(-kappa_uvskin*x)
  etag(0)=exp(-kglob_extinct*x)
  etag(11)=exp(-kglob_uvskin*x)
!  write(*,*) 'h,hr,a =',h,hr,a
  DO i=1,5
     eta(i)=exp(-kappa_ubvri(i)*x)
     eta(5+i)=exp(-kappa_ugriz(i)*x)
     etag(i)=exp(-kglob_ubvri(i)*x)
     etag(5+i)=exp(-kglob_ugriz(i)*x)
  ENDDO
  DO i=0,11
     etad(i)=etag(i)-eta(i)
  ENDDO
END SUBROUTINE calc_transmission

SUBROUTINE calc_colortemp(x,mode,diffmired,cct0,cct)
! Apparent correlated colour temperature (CCT) of the Sun disc
! (averaged over the whole disc) as a function of the air mass x.
! The approximation uses a polynomial fit generated with gnuplot
! (fitted quantity is the difference in the mired scale).
! Source data is ASTMG173 (direct+circumsolar) and the CIE 1931
! colour system. The 4th-power fit is numerically accurate within +2%/-1%
! or +80/-20 K for xair <= 40. Below x=50 the errors are less than
! +2%/-4% resp. +80/-40 K.
! The 5th-power fit is accurate within +5/-1 K = +0.9%/-0.5% below 40
! and with a worst error at x=50 of -20 K or 2%.
! From ASTMG173 the apparent CCT of the extrasolar Sun is found
! to be about 5930 K (i.e. larger than the effectiv temperature
! of 5780 K).
! Note that changing weather conditions may cause MUCH larger errors
! than the numerical residuals.
  IMPLICIT NONE
  INTEGER :: mode
  REAL(KIND=8) :: x,diffmired,mired,cct
  REAL(KIND=8) :: cct0,mired0
!  4th degree polynomial fit coefficients
!  REAL(KIND=8) :: a=14.8639d0,b=-0.258734d0,c=0.0122203d0,d=-0.00011892d0
!  5th degree polynomial fit coefficients
  REAL(KIND=8) :: a,b,c,d,e,f,g,h
  REAL(KIND=8) :: a1=1.55033989D+01, b1=-4.08352380D-01, c1=2.34092449D-02, &
                      d1=-4.53261062D-04,e1=3.46745215D-06
  REAL(KIND=8) :: a2=6.29518119D+00, b2=2.71058584D+00, c2=-4.85304376D-01, &
                      d2=4.18893656D-02, e2=-2.00491235D-03,f2=5.22729222D-05, &
                      g2=-6.93574089D-07,h2=3.66880117D-09
  REAL(KIND=8) :: a3=5.59320584D+00, b3=1.39759248D+00, c3=-3.86100337D-01, &
                      d3=3.08203559D-02, e3=-1.20307771D-03,f3=2.51864627D-05, &
                      g3=-2.69896185D-07,h3=1.15116702D-09

  mired0=1.d6/cct0
  IF (x>50.d0) THEN
     diffmired=1.d99
     cct=0.d0
     RETURN
  ENDIF
  IF (mode==1) THEN!direct + 2.5 deg
     a=a1;b=b1;c=c1;d=d1;e=e1;f=0.d0;g=0.d0;h=0.d0
  ELSE IF (mode==2) THEN!global, tilted towards Sun
     a=a2;b=b2;c=c2;d=d2;e=e2;f=f2;g=g2;h=h2
  ELSE IF (mode==3) THEN!global, horizontal surface
     a=a3;b=b3;c=c3;d=d3;e=e3;f=f3;g=g3;h=h3
  ELSE!space
!     a=0.d0;b=0.d0;c=0.d0;d=0.d0;e=0.d0;m=0.d0
     diffmired=0.d0
     cct=cct0
     RETURN
  ENDIF
! 4th degree polynomial fit
!  diffmired=(a+(b+(c+d*x)*x)*x)*x
! 5th degree polynomial fit
!  diffmired=(a+(b+(c+(d+(e+f*x)*x)*x)*x)*x)*x
  diffmired=(a+(b+(c+(d+(e+(f+(g+h*x)*x)*x)*x)*x)*x)*x)*x
  mired=mired0+diffmired
  cct=1.d6/mired
!  if (mode==3) write(*,*) 'x,dmr,mired0,mired,cct =',x,diffmired,mired0,mired,cct
  RETURN
END SUBROUTINE calc_colortemp

SUBROUTINE calc_anomaly(JD,t2000,L,g)
  USE sunpath_constants
  IMPLICIT NONE
  REAL(KIND=8) :: JD,t2000,L,g,r_deg
  t2000=JD-JD2000
  L=(280.460d0+0.9856474d0*t2000)*d2r
  g=(357.528d0+0.9856003d0*t2000)*d2r
END SUBROUTINE calc_anomaly

SUBROUTINE apparent_sun(g,r_deg)
  IMPLICIT NONE
  REAL(KIND=8) g,r_deg,dist_au
  REAL(KIND=8),PARAMETER :: d2r=1.74532925199432957692d-2
  dist_au=1.00014d0-0.01671d0*cos(g)-0.00014d0*cos(2.d0*g)
  r_deg=0.2666d0/dist_au
END SUBROUTINE apparent_sun

SUBROUTINE adjust_angle(a,a1,mode)
!---- mode = 0: set to ]-pi,pi]
!            1: set to [0,2pi[
  USE sunpath_constants
  IMPLICIT NONE
  INTEGER mode,ic
  REAL(KIND=8) x,a,a1
  LOGICAL ganz
  x=a/ci
  IF (mode.le.0) x=x-0.5d0
  ganz=x.EQ.aint(x)
  IF (ganz .OR. x.GE.0.d0) THEN
     ic=aint(x)
  ELSE
     ic=aint(x)-1.D0
  ENDIF
  a1=a-ic*ci
!  IF (mode.eq.0 .and. .not.ganz) THEN
!     a1=a1-pi
!  ENDIF
  IF (mode.le.0 .and. .not. ganz) a1=a1-ci
!  write(*,*) 'mode,a,a1,ganz = ',mode,a,a1,ganz
!  write(*,*) 'a,a1 = ',a,a1
!  write(*,*) 'x,ic,a-ic*ci = ',a,x,ic,a-ic*ci
  RETURN
END SUBROUTINE adjust_angle

FUNCTION ftwilight(h)
! From http://amsglossary.allenpress.com/glossary
! Civil twilight illuminance decreases from ∼585–410 lux to ∼3.5–2 lux
! Natical twilight: from ~3.5-2 lux to ~0.008 lux
! Astronomical twilight: from ∼0.008 lux to ∼0.0006 lux
  USE sunpath_constants, ONLY : d2r
  IMPLICIT NONE
  REAL(KIND=8) :: ftwilight,h,lmin=1.18d-6
  ftwilight=(1.d0-lmin)*2.7d0**min(0.5d0+h,0.d0)+lmin
!  ftwilight=ftwilight*(1.d0+cos(max(min(20.d0*h,9.d1),0.d0)*d2r)**2)
  RETURN
END FUNCTION ftwilight

SUBROUTINE sun_distance(JD,rscale)
! Approximation of the relative Sun-Earth distance
! Used, so far, only for illuminance calculation
  USE sunpath_constants, ONLY : ci
  IMPLICIT NONE
  INTEGER :: irev
  REAL(KIND=8) :: JD,rscale
  REAL(KIND=8) :: dt,x
  REAL(KIND=8) :: JD_peri2011=2455565.3125d0
  REAL(KIND=8) :: yr_sid=365.256363004d0
  dt=JD-JD_peri2011
  irev=floor(dt/yr_sid)
  x=ci*(dt/yr_sid-irev)
  rscale=(1.d0-0.01671d0*cos(x))
!  write(*,*) 'dt,irev,x/days, rscale =',dt,irev,x*365.25d0/ci,rscale
END SUBROUTINE sun_distance
